<?php

defined('BASEPATH') or exit('No direct script access allowed');

/* * *****************Setting.php**********************************
 * @product name    : Hospital Management System
 * @type            : Class
 * @class name      : Setting
 * @description     : Manage application general settings.
 * @author          : South Bangla IT Zone Team
 * @url             : http://hms.sbitzone.com
 * @support         : rabibd.it@gmail.com
 * @copyright       : South Bangla IT Zone Team
 * ********************************************************** */

class Setting extends MY_Controller {

    public $data = array();

    function __construct() {
        parent::__construct();
        $this->load->model('Setting_Model', 'setting', true);
    }

    /*     * ***************Function index**********************************
     * @type            : Function
     * @function name   : index
     * @description     : Load "General Setting" user interface
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */

    public function index() {

        check_permission(VIEW);

        $this->data['setting'] = $this->setting->get_single('global_setting', array('status' => 1));

        $this->layout->title($this->lang->line('general') . ' ' . $this->lang->line('setting') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('setting/index', $this->data);
    }

    /*     * ***************Function support**********************************
     * @type            : Function
     * @function name   : support
     * @description     : Load "General support" user interface
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */

    public function support() {

        check_permission(VIEW);
        $this->layout->title($this->lang->line('support') . ' ' . $this->lang->line('setting') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('setting/support', $this->data);
    }

    /*     * ***************Function add**********************************
     * @type            : Function
     * @function name   : add
     * @description     : Load "New General Settings" user interface
     *                    and process to store "General Settings" into database
     *                    for the first time settings
     * @param           : null
     * @return          : null
     * ********************************************************** */

    public function add() {

        check_permission(ADD);

        if ($_POST) {
            $this->_prepare_setting_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_setting_data();

                $insert_id = $this->setting->insert('global_setting', $data);
                if ($insert_id) {
                    create_log('Has been created global setting');
                    success($this->lang->line('insert_success'));
                    redirect('setting/index');
                } else {
                    error($this->lang->line('insert_failed'));
                    redirect('setting/add');
                }
            } else {
                $this->data = $_POST;
            }
        }

        $this->layout->title($this->lang->line('general') . ' ' . $this->lang->line('setting') . ' | ' . $this->data['setting']->brand_title);
        $this->layout->view('setting/index', $this->data);
    }

    /*     * ***************Function edit**********************************
     * @type            : Function
     * @function name   : edit
     * @description     : Load Update "General Settings" user interface
     *                    with populate "General Settings" value
     *                    and process to update "General Settings" into database
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */

    public function edit($id = null) {

        check_permission(EDIT);

        if ($_POST) {
            $this->_prepare_setting_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_setting_data();
                $updated = $this->setting->update('global_setting', $data, array('id' => $this->input->post('id')));

                if ($updated) {
                    create_log('Has been updated global setting');
                    success($this->lang->line('update_success'));
                    redirect('setting/index');
                } else {
                    error($this->lang->line('update_failed'));
                    redirect('setting/edit/' . $this->input->post('id'));
                }
            }
        }

        $this->layout->title($this->lang->line('general') . ' ' . $this->lang->line('setting') . ' | ' . $this->data['setting']->brand_title ? $this->data['setting']->brand_title : TITLE);
        $this->layout->view('setting/index', $this->data);
    }

    /*     * ***************Function _prepare_setting_validation**********************************
     * @type            : Function
     * @function name   : _prepare_setting_validation
     * @description     : Process "General Settings" user input data validation
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */

    private function _prepare_setting_validation() {

        $this->load->library('form_validation');
        $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');

        $this->form_validation->set_rules('brand_title', $this->lang->line('brand') . ' ' . $this->lang->line('title'), 'trim');
        $this->form_validation->set_rules('brand_name', $this->lang->line('brand') . ' ' . $this->lang->line('name'), 'trim');
        $this->form_validation->set_rules('time_zone', $this->lang->line('time_zone'), 'trim|required');
        $this->form_validation->set_rules('date_format', $this->lang->line('date_format'), 'trim|required');
        $this->form_validation->set_rules('time_format', $this->lang->line('time_format'), 'trim|required');
        $this->form_validation->set_rules('phone', $this->lang->line('phone'), 'trim|required');
        $this->form_validation->set_rules('email', $this->lang->line('email'), 'trim|required');
        $this->form_validation->set_rules('brand_footer', $this->lang->line('brand') . ' ' . $this->lang->line('footer'), 'trim');
    }

    /*     * ***************Function _get_posted_setting_data**********************************
     * @type            : Function
     * @function name   : _get_posted_setting_data
     * @description     : Prepare "General Settings" user input data to save into database
     *
     * @param           : null
     * @return          : $data array(); value
     * ********************************************************** */

    private function _get_posted_setting_data() {

        $items = array();
        $items[] = 'brand_title';
        $items[] = 'brand_name';
        $items[] = 'time_zone';
        $items[] = 'date_format';
        $items[] = 'time_format';
        $items[] = 'is_manager_discount';
        $items[] = 'is_discount';
        $items[] = 'address';
        $items[] = 'phone';
        $items[] = 'email';
        $items[] = 'fax';
        $items[] = 'currency';
        $items[] = 'currency_symbol';
        $items[] = 'lat';
        $items[] = 'lng';
        $items[] = 'facebook_url';
        $items[] = 'twitter_url';
        $items[] = 'linkedin_url';
        $items[] = 'youtube_url';
        $items[] = 'instagram_url';
        $items[] = 'skype';
        $items[] = 'brand_footer';
        $items[] = 'video_url';
        $items[] = 'map';

        $data = elements($items, $_POST);

        if ($this->input->post('id')) {
            $data['modified_at'] = date('Y-m-d H:i:s');
            $data['modified_by'] = logged_in_user_id();
        } else {

            $data['status'] = 1;
            $data['created_at'] = date('Y-m-d H:i:s');
            $data['created_by'] = logged_in_user_id();
        }

        if ($_FILES['brand_logo']['name']) {
            $data['brand_logo'] = $this->_upload_brand_logo();
        }
        if ($_FILES['favicon']['name']) {
            $data['favicon'] = $this->_upload_favicon();
        }
        
        if ($_FILES['video_thumbnail']['name']) {
            $data['video_thumbnail'] = $this->_upload_video_thumbnail();
        }

        return $data;
    }

    /*     * ***************Function _upload_brand_logo**********************************
     * @type            : Function
     * @function name   : _upload_brand_logo
     * @description     : Process to upload institute logo in the server
     *                     and return logo name
     * @param           : null
     * @return          : $logo string value
     * ********************************************************** */

    private function _upload_brand_logo() {

        $prev_brand_logo = $this->input->post('prev_brand_logo');
        $brand_logo = $_FILES['brand_logo']['name'];
        $brand_logo_type = $_FILES['brand_logo']['type'];
        $return_brand_logo = '';
        if ($brand_logo != "") {
            if (
                $brand_logo_type == 'image/jpg' || $brand_logo_type == 'image/jpeg' ||
                $brand_logo_type == 'image/jpg' || $brand_logo_type == 'image/png' ||
                $brand_logo_type == 'image/x-png' || $brand_logo_type == 'image/gif'
            ) {

                $destination = 'assets/uploads/logo/';

                $file_type = explode(".", $brand_logo);
                $extension = strtolower($file_type[count($file_type) - 1]);
                $brand_logo_path = 'brand_logo-' . time() . '-sbiz.' . $extension;

                move_uploaded_file($_FILES['brand_logo']['tmp_name'], $destination . $brand_logo_path);

                // need to unlink previous brand_logo
                if ($prev_brand_logo != "") {
                    if (file_exists($destination . $prev_brand_logo)) {
                        @unlink($destination . $prev_brand_logo);
                    }
                }

                $return_brand_logo = $brand_logo_path;
            }
        } else {
            $return_brand_logo = $prev_brand_logo;
        }

        return $return_brand_logo;
    }


    /*****************Function _upload_favicon**********************************
     * @type            : Function
     * @function name   : _upload_favicon
     * @description     : Process to upload institute front favicon in the server
     *                     and return favicon name
     * @param           : null
     * @return          : $favicon string value
     * ********************************************************** */
    private function _upload_favicon() {

        $prev_favicon = $this->input->post('prev_favicon');
        $favicon = $_FILES['favicon']['name'];
        $favicon_type = $_FILES['favicon']['type'];
        $return_favicon = '';
        if ($favicon != "") {
            if (
                $favicon_type == 'image/jpeg' || $favicon_type == 'image/pjpeg' ||
                $favicon_type == 'image/jpg' || $favicon_type == 'image/png' ||
                $favicon_type == 'image/x-png' || $favicon_type == 'image/gif'
            ) {

                $destination = 'assets/uploads/icon/';

                $file_type = explode(".", $favicon);
                $extension = strtolower($file_type[count($file_type) - 1]);
                $favicon_path = 'favicon-' . time() . '-sbiz.' . $extension;

                move_uploaded_file($_FILES['favicon']['tmp_name'], $destination . $favicon_path);

                // need to unlink previous favicon
                if ($prev_favicon != "") {
                    if (file_exists($destination . $prev_favicon)) {
                        @unlink($destination . $prev_favicon);
                    }
                }

                $return_favicon = $favicon_path;
            }
        } else {
            $return_favicon = $prev_favicon;
        }

        return $return_favicon;
    }
    
    /*     * ***************Function _upload_video_thumbnail**********************************
     * @type            : Function
     * @function name   : _upload_video_thumbnail
     * @description     : Process to upload institute logo in the server
     *                     and return logo name
     * @param           : null
     * @return          : $logo string value
     * ********************************************************** */

    private function _upload_video_thumbnail() {

        $prev_video_thumbnail = $this->input->post('prev_video_thumbnail');
        $video_thumbnail = $_FILES['video_thumbnail']['name'];
        $video_thumbnail_type = $_FILES['video_thumbnail']['type'];
        $return_video_thumbnail = '';
        if ($video_thumbnail != "") {
            if (
                $video_thumbnail_type == 'image/jpg' || $video_thumbnail_type == 'image/jpeg' ||
                $video_thumbnail_type == 'image/jpg' || $video_thumbnail_type == 'image/png' ||
                $video_thumbnail_type == 'image/x-png' || $video_thumbnail_type == 'image/gif'
            ) {

                $destination = 'assets/uploads/logo/';

                $file_type = explode(".", $video_thumbnail);
                $extension = strtolower($file_type[count($file_type) - 1]);
                $video_thumbnail_path = 'video_thumbnail-' . time() . '-sbiz.' . $extension;

                move_uploaded_file($_FILES['video_thumbnail']['tmp_name'], $destination . $video_thumbnail_path);

                // need to unlink previous video_thumbnail
                if ($prev_video_thumbnail != "") {
                    if (file_exists($destination . $prev_video_thumbnail)) {
                        @unlink($destination . $prev_video_thumbnail);
                    }
                }

                $return_video_thumbnail = $video_thumbnail_path;
            }
        } else {
            $return_video_thumbnail = $prev_video_thumbnail;
        }

        return $return_video_thumbnail;
    }
}
