<?php

defined('BASEPATH') or exit('No direct script access allowed');

/* * *****************Custom.php**********************************
 * @product name    : Hospital Management System
 * @type            : Class
 * @class name      : Custom
 * @description     : Manage system user custom permission.
 * @author          : South Bangla IT Zone Team
 * @url             : https://hms.sbitzone.com
 * @support         : rabibd.it@gmail.com
 * @copyright       : South Bangla IT Zone Team
 * ********************************************************** */

class Custom extends MY_Controller {

    public $data = array();

    function __construct() {
        parent::__construct();
        $this->load->model('Setting_Model', 'setting', true);
    }

    /*****************Function index**********************************
     * @type            : Function
     * @function name   : index
     * @description     : Load "User Permisson List" user interface
     *
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function index() {

        check_permission(VIEW);

        $search = $this->input->get('search');
        $config = array();
        $config = get_pagination(ADMIN_PER_PAGE);
        if ($search) {
            $config['base_url'] = site_url('setting/custom/?search=') . $search;
        } else {
            $config['base_url'] = current_url();
        }
        if ($this->input->get('page')) {
            $sgm = (int) trim($this->input->get('page'));
            $offset = $config['per_page'] * ($sgm - 1);
        } else {
            $offset = 0;
        }

        $config['total_rows'] = $this->setting->get_custom_permission_list($config['per_page'], $offset, $search, $count = TRUE);
        $this->pagination->initialize($config);
        $this->data['permissions'] = $this->setting->get_custom_permission_list($config['per_page'], $offset, $search, $count = FALSE);
        $this->data['sl_count'] = ($this->input->get('page') == 0 ? 1 : (($this->input->get('page') - 1) * $config["per_page"] + 1));
        $this->data['search'] = $search;

        $this->data['roles'] = $this->setting->get_list('roles', array('status' => 1), '', '', '', 'id', 'ASC');
        $this->data['list'] = TRUE;
        $this->layout->title($this->lang->line('permission') . ' | ' .  $this->global_setting->brand_title);
        $this->layout->view('custom/index', $this->data);
    }

    /*****************Function add**********************************
     * @type            : Function
     * @function name   : add
     * @description     : Load "Add new Permission" user interface
     *                    and store "permissions" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function add() {

        check_permission(ADD);

        if ($_POST) {
            $this->_prepare_custom_permission_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_custom_permission_data();
                $insert_id = $this->setting->insert('permissions', $data);
                if ($insert_id) {
                    $user = $this->setting->get_single('employees', array('user_id' => $data['user_id']));
                    create_log('Has been created a user custom permission : ' . $user->name . ' =>' . $data['permission_type']);
                    success($this->lang->line('insert_success'));
                    redirect('setting/custom');
                } else {
                    error($this->lang->line('insert_failed'));
                    redirect('setting/custom/add');
                }
            } else {
                $this->data = $_POST;
            }
        }

        $this->data['roles'] = $this->setting->get_list('roles', array('status' => 1), '', '', '', 'id', 'ASC');
        $this->data['add'] = TRUE;
        $this->layout->title($this->lang->line('add') . ' ' . $this->lang->line('permission') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('custom/add', $this->data);
    }

    /*****************Function edit**********************************
     * @type            : Function
     * @function name   : edit
     * @description     : Load Update "permissions" user interface
     *                    with populated "permissions" value
     *                    and update "permissions" database
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function edit($id = null) {

        check_permission(EDIT);

        if ($_POST) {
            $this->_prepare_custom_permission_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_custom_permission_data();
                $updated = $this->setting->update('permissions', $data, array('id' => $this->input->post('id')));
                if ($updated) {
                    $user = $this->setting->get_single('employees', array('user_id' => $data['user_id']));
                    create_log('Has been updated a user custom permission : ' . $user->name . ' =>' . $data['permission_type']);
                    success($this->lang->line('update_success'));
                    redirect('setting/custom');
                } else {
                    error($this->lang->line('update_failed'));
                    redirect('setting/custom/edit/' . $this->input->post('id'));
                }
            } else {
                $this->data['permission'] = $this->setting->get_single('permissions', array('id' => $this->input->post('id')));
            }
        } else {
            if ($id) {
                $this->data['permission'] = $this->setting->get_single('permissions', array('id' => $id));
                if (!$this->data['permission']) {
                    redirect('setting/custom');
                }
            }
        }

        $this->data['roles'] = $this->setting->get_list('roles', array('status' => 1), '', '', '', 'id', 'ASC');
        $this->data['users'] = $this->setting->get_list('employees', array('status' => 1), '', '', '', 'id', 'ASC');
        $this->data['edit'] = TRUE;
        $this->layout->title($this->lang->line('edit') . ' ' . $this->lang->line('permission') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('custom/edit', $this->data);
    }

    /*****************Function _prepare_custom_permission_validation**********************************
     * @type            : Function
     * @function name   : _prepare_custom_permission_validation
     * @description     : Process "permission" user input data validation
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    private function _prepare_custom_permission_validation() {
        $this->load->library('form_validation');
        $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
        $this->form_validation->set_rules('permission_type', $this->lang->line('permission') . ' ' . $this->lang->line('type'), 'trim|required|callback_type');
        $this->form_validation->set_rules('role_id', $this->lang->line('role') . ' ' . $this->lang->line('name'), 'trim|required');
        $this->form_validation->set_rules('user_id', $this->lang->line('user') . ' ' . $this->lang->line('name'), 'trim|required');
        $this->form_validation->set_rules('note', $this->lang->line('note'), 'trim');
    }

    /*****************Function type**********************************
     * @type            : Function
     * @function name   : type
     * @description     : Unique check for "Permission Type" data/value
     *
     * @param           : null
     * @return          : boolean true/false
     * ********************************************************** */
    public function type() {
        if ($this->input->post('id') == '') {
            $type = $this->setting->duplicate_check_custom_permission($this->input->post('role_id'), $this->input->post('user_id'), $this->input->post('permission_type'));
            if ($type) {
                $this->form_validation->set_message('type', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else if ($this->input->post('id') != '') {
            $type = $this->setting->duplicate_check_custom_permission($this->input->post('role_id'), $this->input->post('user_id'), $this->input->post('permission_type'), $this->input->post('id'));
            if ($type) {
                $this->form_validation->set_message('type', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else {
            return TRUE;
        }
    }

    /*****************Function _get_posted_custom_permission_data**********************************
     * @type            : Function
     * @function name   : _get_posted_custom_permission_data
     * @description     : Prepare "Custom Permission" user input data to save into database
     *
     * @param           : null
     * @return          : $data array(); value
     * ********************************************************** */
    private function _get_posted_custom_permission_data() {

        $items = array();
        $items[] = 'role_id';
        $items[] = 'user_id';
        $items[] = 'permission_type';
        $items[] = 'note';

        $data = elements($items, $_POST);
        $data['permissions_status'] = $this->input->post('permissions_status') ?? 0;

        if ($this->input->post('id')) {
            $data['modified_at'] = date('Y-m-d H:i:s');
            $data['modified_by'] = logged_in_user_id();
        } else {
            $data['created_at'] = date('Y-m-d H:i:s');
            $data['created_by'] = logged_in_user_id();
        }

        return $data;
    }

    /*****************Function delete**********************************
     * @type            : Function
     * @function name   : delete
     * @description     : delete "Permission" from database
     *
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function delete($id = null) {

        check_permission(DELETE);
        if (!is_numeric($id)) {
            error($this->lang->line('unexpected_error'));
            redirect('setting/custom');
        }

        $permission = $this->setting->get_single('permissions', array('id' => $id));
        if ($this->setting->delete('permissions', array('id' => $id))) {
            $user = $this->setting->get_single('employees', array('user_id' => $permission->user_id));
            create_log('Has been deleted a user custom permission : ' . $user->name . ' =>' . $permission->permission_type);
            success($this->lang->line('delete_success'));
        } else {
            error($this->lang->line('delete_failed'));
        }
        redirect($_SERVER['HTTP_REFERER']);
    }
}
