<?php

defined('BASEPATH') or exit('No direct script access allowed');

/* * *****************Receive.php**********************************
 * @product name    : Hospital Management System
 * @type            : Class
 * @class name      : Receive
 * @description     : Manage receive.
 * @author          : South Bangla IT Zone Team
 * @url             : http://hms.sbitzone.com
 * @support         : rabibd.it@gmail.com
 * @copyright       : South Bangla IT Zone Team
 * ********************************************************** */

class Receive extends MY_Controller {

    public $data = array();

    function __construct() {
        parent::__construct();
        $this->load->model('Receive_Model', 'receive', true);
    }

    /*****************Function index**********************************
     * @type            : Function
     * @function name   : index
     * @description     : Load "Receive List" user interface
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function index() {

        check_permission(VIEW);
        
        $search = $this->input->get('search');
        $config = array();
        $config = get_pagination(ADMIN_PER_PAGE);
        if ($search) {
            $config['base_url'] = site_url('frontoffice/receive/?search=') . $search;
        } else {
            $config['base_url'] = current_url();
        }
        if ($this->input->get('page')) {
            $sgm = (int) trim($this->input->get('page'));
            $offset = $config['per_page'] * ($sgm - 1);
        } else {
            $offset = 0;
        }

        $config['total_rows'] = $this->receive->get_receive_list($config['per_page'], $offset, $search, $count = TRUE);
        $this->pagination->initialize($config);
        $this->data['receives'] = $this->receive->get_receive_list($config['per_page'], $offset, $search, $count = FALSE);

        $this->data['sl_count'] = ($this->input->get('page') == 0 ? 1 : (($this->input->get('page') - 1) * $config["per_page"] + 1));
        $this->data['search'] = $search;

        $this->data['list'] = TRUE;
        $this->layout->title($this->lang->line('postal_receive') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('receive/index', $this->data);
    }

    /*****************Function add**********************************
     * @type            : Function
     * @function name   : add
     * @description     : Load "Add new Receive" user interface
     *                    and process to store "receives" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function add() {

        check_permission(ADD);
        if ($_POST) {
            $this->_prepare_receive_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_receive_data();
                $insert_id = $this->receive->insert('receives', $data);
                if ($insert_id) {
                    create_log('Has been created a Receive : ' . $data['from_title']);
                    success($this->lang->line('insert_success'));
                    redirect('frontoffice/receive');
                } else {
                    error($this->lang->line('insert_failed'));
                    redirect('frontoffice/receive/add');
                }
            } else {
                $this->data = $_POST;
            }
        }

        $this->data['add'] = TRUE;
        $this->layout->title($this->lang->line('add') . ' ' . $this->lang->line('receive') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('receive/add', $this->data);
    }

    /*****************Function edit**********************************
     * @type            : Function
     * @function name   : edit
     * @description     : Load Update "Receive" user interface
     *                    with populate "Receive" value
     *                    and process to update "receives" into database
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function edit($id = null) {

        check_permission(EDIT);
        if ($_POST) {
            $this->_prepare_receive_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_receive_data();
                $updated = $this->receive->update('receives', $data, array('id' => $this->input->post('id')));
                if ($updated) {
                    create_log('Has been updated a Receive : ' . $data['from_title']);
                    success($this->lang->line('update_success'));
                    redirect('frontoffice/receive');
                } else {
                    error($this->lang->line('update_failed'));
                    redirect('frontoffice/receive/edit/' . $this->input->post('id'));
                }
            } else {
                $this->data['receive'] = $this->receive->get_single('receives', array('id' => $this->input->post('id')));
            }
        } else {
            if ($id) {
                $this->data['receive'] = $this->receive->get_single('receives', array('id' => $id));
                if (!$this->data['receive']) {
                    redirect('frontoffice/receive');
                }
            }
        }

        $this->data['edit'] = TRUE;
        $this->layout->title($this->lang->line('edit') . ' ' . $this->lang->line('receive') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('receive/edit', $this->data);
    }

    /*****************Function _prepare_receive_validation**********************************
     * @type            : Function
     * @function name   : _prepare_receive_validation
     * @description     : Process "Receive" user input data validation
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    private function _prepare_receive_validation() {
        $this->load->library('form_validation');
        $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
        $this->form_validation->set_rules('from_title', $this->lang->line('from_title'), 'trim|required');
        $this->form_validation->set_rules('date', $this->lang->line('date'), 'trim|required');
    }

    /*****************Function _get_posted_receive_data**********************************
     * @type            : Function
     * @function name   : _get_posted_receive_data
     * @description     : Prepare "Receive" user input data to save into database
     *
     * @param           : null
     * @return          : $data array(); value
     * ********************************************************** */
    private function _get_posted_receive_data() {

        $items = array();
        $items[] = 'from_title';
        $items[] = 'reference';
        $items[] = 'to_title';
        $items[] = 'address';
        $items[] = 'note';
        $data = elements($items, $_POST);

        $data['date'] = $this->input->post('date') ? date('Y-m-d', strtotime($this->input->post('date'))) : date('Y-m-d');

        if ($this->input->post('id')) {
            $data['modified_at'] = date('Y-m-d H:i:s');
            $data['modified_by'] = logged_in_user_id();
        } else {
            $data['status'] = 1;
            $data['created_at'] = date('Y-m-d H:i:s');
            $data['created_by'] = logged_in_user_id();
        }

        if ($_FILES['attachment']['name']) {
            $data['attachment'] = $this->_upload_attachment();
        }

        return $data;
    }

    /*     * ***************Function _upload_attachment**********************************
     * @type            : Function
     * @function name   : _upload_attachment
     * @description     : Process to upload receive attachment into server
     *                     and return attachment file name
     * @param           : null
     * @return          : $return_attachment string value
     * ********************************************************** */

    private function _upload_attachment() {

        $prev_attachment = $this->input->post('prev_attachment');
        $attachment = $_FILES['attachment']['name'];
        $attachment_type = $_FILES['attachment']['type'];
        $return_attachment = '';

        if ($attachment != "") {
            if (
                $attachment_type == 'application/vnd.openxmlformats-officedocument.wordprocessingml.document' ||
                $attachment_type == 'application/msword' || $attachment_type == 'text/plain' ||
                $attachment_type == 'application/vnd.ms-office' || $attachment_type == 'application/pdf' ||
                $attachment_type == 'image/jpeg' || $attachment_type == 'image/pjpeg' ||
                $attachment_type == 'image/jpg' || $attachment_type == 'image/png' ||
                $attachment_type == 'image/x-png' || $attachment_type == 'image/gif'
            ) {

                $destination = 'assets/uploads/attachment/';

                $file_type = explode(".", $attachment);
                $extension = strtolower($file_type[count($file_type) - 1]);
                $attachment_path = 'receive-attachment-' . time() . '-sbiz.' . $extension;

                move_uploaded_file($_FILES['attachment']['tmp_name'], $destination . $attachment_path);

                // need to unlink previous photo
                if ($prev_attachment != "") {
                    if (file_exists($destination . $prev_attachment)) {
                        @unlink($destination . $prev_attachment);
                    }
                }

                $return_attachment = $attachment_path;
            }
        } else {
            $return_attachment = $prev_attachment;
        }

        return $return_attachment;
    }

    /*****************Function delete**********************************
     * @type            : Function
     * @function name   : delete
     * @description     : delete "Receive" data from database
     *
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function delete($id = null) {

        check_permission(DELETE);

        if (!is_numeric($id)) {
            error($this->lang->line('unexpected_error'));
            redirect('frontoffice/receive');
        }

        $receive = $this->receive->get_single('receives', array('id' => $id));

        if ($this->receive->delete('receives', array('id' => $id))) {
            // delete attachment
            $destination = 'assets/uploads/';
            if (file_exists($destination . '/attachment/' . $receive->attachment)) {
                @unlink($destination . '/attachment/' . $receive->attachment);
            }
            create_log('Has been deleted a Receive : ' . $receive->from_title);
            success($this->lang->line('delete_success'));
        } else {
            error($this->lang->line('delete_failed'));
        }
        redirect('frontoffice/receive');
    }
}
