<?php

defined('BASEPATH') or exit('No direct script access allowed');

/* * *****************Schedule.php**********************************
 * @product name    : Hospital Management System
 * @type            : Class
 * @class name      : Schedule
 * @description     : Manage schedule.
 * @author          : South Bangla IT Zone Team
 * @url             : http://hms.sbitzone.com
 * @support         : rabibd.it@gmail.com
 * @copyright       : South Bangla IT Zone Team
 * ********************************************************** */

class Schedule extends MY_Controller {

    public $data = array();

    function __construct() {
        parent::__construct();
        $this->load->model('Schedule_Model', 'schedule', true);
    }

    /*****************Function index**********************************
     * @type            : Function
     * @function name   : index
     * @description     : Load "Schedule List" user interface
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function index() {

        check_permission(VIEW);

        $search = trim($this->input->get('search'));

        $config = array();
        $config = get_pagination(ADMIN_PER_PAGE);
        if ($search) {
            $config['base_url'] = site_url('doctor/schedule/?search=') . $search;
        } else {
            $config['base_url'] = current_url();
        }
        if ($this->input->get('page')) {
            $sgm = (int) trim($this->input->get('page'));
            $offset = $config['per_page'] * ($sgm - 1);
        } else {
            $offset = 0;
        }

        $config['total_rows'] = $this->schedule->get_schedule_list($config['per_page'], $offset, $search, $count = TRUE);
        $this->pagination->initialize($config);
        $this->data['schedules'] = $this->schedule->get_schedule_list($config['per_page'], $offset, $search, $count = FALSE);
        $this->data['sl_count'] = ($this->input->get('page') == 0 ? 1 : (($this->input->get('page') - 1) * $config["per_page"] + 1));
        $this->data['search'] = $search;

        $this->data['list'] = TRUE;
        $this->layout->title($this->lang->line('manage_schedule') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('schedule/index', $this->data);
    }

    /*****************Function add**********************************
     * @type            : Function
     * @function name   : add
     * @description     : Load "Add new Schedule" user interface
     *                    and process to store "schedules" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function add() {

        check_permission(ADD);
        if ($_POST) {
            $this->_prepare_schedule_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_schedule_data();
                $insert_id = $this->schedule->insert('doctor_schedules', $data);
                if ($insert_id) {
                    create_log('Has been created a Schedule : ' . $data['day']);
                    success($this->lang->line('insert_success'));
                    redirect('doctor/schedule');
                } else {
                    error($this->lang->line('insert_failed'));
                    redirect('doctor/schedule/add');
                }
            } else {
                $this->data = $_POST;
            }
        }
        $this->data['doctors'] = $this->schedule->get_doctor_list();
        $this->data['add'] = TRUE;
        $this->layout->title($this->lang->line('add') . ' ' . $this->lang->line('schedule') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('schedule/add', $this->data);
    }

    /*****************Function edit**********************************
     * @type            : Function
     * @function name   : edit
     * @description     : Load Update "Schedule" user interface
     *                    with populate "Schedule" value
     *                    and process to update "schedules" into database
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function edit($id = null) {

        check_permission(EDIT);
        if ($_POST) {
            $this->_prepare_schedule_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_schedule_data();
                $updated = $this->schedule->update('doctor_schedules', $data, array('id' => $this->input->post('id')));
                if ($updated) {
                    create_log('Has been updated a Schedule : ' . $data['name']);
                    success($this->lang->line('update_success'));
                    redirect('doctor/schedule');
                } else {
                    error($this->lang->line('update_failed'));
                    redirect('doctor/schedule/edit/' . $this->input->post('id'));
                }
            } else {
                $this->data['schedule'] = $this->schedule->get_single('doctor_schedules', array('id' => $this->input->post('id')));
            }
        } else {
            if ($id) {
                $this->data['schedule'] = $this->schedule->get_single('doctor_schedules', array('id' => $id));
                if (!$this->data['schedule']) {
                    redirect('doctor/schedule');
                }
            }
        }
        $this->data['doctors'] = $this->schedule->get_doctor_list();
        $this->data['edit'] = TRUE;
        $this->layout->title($this->lang->line('edit') . ' ' . $this->lang->line('schedule') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('schedule/edit', $this->data);
    }

    /*****************Function _prepare_schedule_validation**********************************
     * @type            : Function
     * @function name   : _prepare_schedule_validation
     * @description     : Process "Schedule" user input data validation
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    private function _prepare_schedule_validation() {
        $this->load->library('form_validation');
        $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');

        $this->form_validation->set_rules('doctor_id', $this->lang->line('doctor') . ' ' . $this->lang->line('name'), 'trim|required');
        $this->form_validation->set_rules('day', $this->lang->line('schedule') . ' ' . $this->lang->line('day'), 'trim|required|callback_day');
        $this->form_validation->set_rules('start_time', $this->lang->line('start_time'), 'trim|required');
        $this->form_validation->set_rules('end_time', $this->lang->line('end_time'), 'trim|required');
    }

    /*****************Function day**********************************
     * @type            : Function
     * @function name   : day
     * @description     : Unique check for "Schedule Day" data/value
     *
     * @param           : null
     * @return          : boolean true/false
     * ********************************************************** */
    public function day() {
        if ($this->input->post('id') == '') {
            $schedule = $this->schedule->duplicate_check($this->input->post('doctor_id'), $this->input->post('day'));
            if ($schedule) {
                $this->form_validation->set_message('day', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else if ($this->input->post('id') != '') {
            $schedule = $this->schedule->duplicate_check($this->input->post('doctor_id'), $this->input->post('day'), $this->input->post('id'));
            if ($schedule) {
                $this->form_validation->set_message('day', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else {
            return TRUE;
        }
    }

    /*****************Function _get_posted_schedule_data**********************************
     * @type            : Function
     * @function name   : _get_posted_schedule_data
     * @description     : Prepare "Schedule" user input data to save into database
     *
     * @param           : null
     * @return          : $data array(); value
     * ********************************************************** */
    private function _get_posted_schedule_data() {

        $items = array();
        $items[] = 'doctor_id';
        $items[] = 'day';
        $items[] = 'per_patient_time';
        $items[] = 'serial_visibility';
        $items[] = 'note';
        $items[] = 'sort_by';
        $items[] = 'status';
        $data = elements($items, $_POST);

        $data['start_time'] = date('H:i:s', strtotime($this->input->post('start_time')));
        $data['end_time'] = date('H:i:s', strtotime($this->input->post('end_time')));

        if ($this->input->post('id')) {
            $data['modified_at'] = date('Y-m-d H:i:s');
            $data['modified_by'] = logged_in_user_id();
        } else {
            $data['created_at'] = date('Y-m-d H:i:s');
            $data['created_by'] = logged_in_user_id();
        }
        return $data;
    }

    /*****************Function delete**********************************
     * @type            : Function
     * @function name   : delete
     * @description     : delete "Schedule" data from database
     *
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function delete($id = null) {

        check_permission(DELETE);

        if (!is_numeric($id)) {
            error($this->lang->line('unexpected_error'));
            redirect('doctor/schedule');
        }

        $schedule = $this->schedule->get_single('doctor_schedules', array('id' => $id));

        if ($this->schedule->delete('doctor_schedules', array('id' => $id))) {
            create_log('Has been deleted a Schedule : ' . $schedule->day);
            success($this->lang->line('delete_success'));
        } else {
            error($this->lang->line('delete_failed'));
        }
        redirect($_SERVER['HTTP_REFERER']);
    }
}
