<?php

defined('BASEPATH') or exit('No direct script access allowed');

/* * *****************Holiday.php**********************************
 * @product name    : Hospital Management System
 * @type            : Class
 * @class name      : Holiday
 * @description     : Manage holiday.
 * @author          : South Bangla IT Zone Team
 * @url             : http://hms.sbitzone.com
 * @support         : rabibd.it@gmail.com
 * @copyright       : South Bangla IT Zone Team
 * ********************************************************** */

class Holiday extends MY_Controller {

    public $data = array();

    function __construct() {
        parent::__construct();
        $this->load->model('Holiday_Model', 'holiday', true);
    }

    /*****************Function index**********************************
     * @type            : Function
     * @function name   : index
     * @description     : Load "Holiday List" user interface
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function index() {

        check_permission(VIEW);

        if ($_GET && $this->input->get('date')) {
            $keyword = trim($this->input->get('keyword'));
            $date = $this->input->get('date');
            $exp = explode('-', $date);
            $form_date = date('Y-m-d', strtotime($exp[0]));
            $to_date = date('Y-m-d', strtotime($exp[1]));
            $created_by = $this->input->get('created_by');

            $config = array();
            $config = get_pagination(ADMIN_PER_PAGE);
            $config['base_url'] = site_url('hrm/holiday?keyword=' . $keyword . '&date=' . $date . '&created_by=' . $created_by);
            if ($this->input->get('page')) {
                $sgm = (int) trim($this->input->get('page'));
                $offset = $config['per_page'] * ($sgm - 1);
            } else {
                $offset = 0;
            }
            $config['total_rows'] = $this->holiday->get_holiday_list($config['per_page'], $offset, $keyword, $form_date, $to_date,  $created_by, $count = TRUE);
            $this->pagination->initialize($config);
            $this->data['holidays'] = $this->holiday->get_holiday_list($config['per_page'], $offset, $keyword, $form_date, $to_date,  $created_by, $count = FALSE);
            $this->data['sl_count'] = ($this->input->get('page') == 0 ? 1 : (($this->input->get('page') - 1) * $config["per_page"] + 1));

            $this->data['keyword'] = $keyword;
            $this->data['date'] = $date;
            $this->data['created_by'] = $created_by;
        } else {
            $config = array();
            $config = get_pagination(ADMIN_PER_PAGE);
            $config['base_url'] = current_url();
            if ($this->input->get('page')) {
                $sgm = (int) trim($this->input->get('page'));
                $offset = $config['per_page'] * ($sgm - 1);
            } else {
                $offset = 0;
            }

            $config['total_rows'] = $this->holiday->get_holiday_list($config['per_page'], $offset, null, null, null, null, $count = TRUE);
            $this->pagination->initialize($config);
            $this->data['holidays'] = $this->holiday->get_holiday_list($config['per_page'], $offset, null, null, null, null, $count = FALSE);
            $this->data['sl_count'] = ($this->input->get('page') == 0 ? 1 : (($this->input->get('page') - 1) * $config["per_page"] + 1));
        }

        $this->data['employees'] = $this->holiday->get_employee_list();
        $this->data['list'] = TRUE;
        $this->layout->title($this->lang->line('manage_holiday') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('holiday/index', $this->data);
    }

    /*****************Function add**********************************
     * @type            : Function
     * @function name   : add
     * @description     : Load "Add new Holiday" user interface
     *                    and process to store "holidays" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function add() {

        check_permission(ADD);
        if ($_POST) {
            $this->_prepare_holiday_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_holiday_data();
                $insert_id = $this->holiday->insert('holidays', $data);
                if ($insert_id) {
                    create_log('Has been created a holiday : ' . $data['title']);
                    success($this->lang->line('insert_success'));
                    redirect('hrm/holiday');
                } else {
                    error($this->lang->line('insert_failed'));
                    redirect('hrm/holiday/add');
                }
            } else {
                $this->data = $_POST;
            }
        }
        $this->data['add'] = TRUE;
        $this->layout->title($this->lang->line('add') . ' ' . $this->global_setting->brand_title);
        $this->layout->view('holiday/add', $this->data);
    }

    /*****************Function edit**********************************
     * @type            : Function
     * @function name   : edit
     * @description     : Load Update "Holiday" user interface
     *                    with populate "Holiday" value
     *                    and process to update "holidays" into database
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function edit($id = null) {

        check_permission(EDIT);
        if ($_POST) {
            $this->_prepare_holiday_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_holiday_data();
                $updated = $this->holiday->update('holidays', $data, array('id' => $this->input->post('id')));
                if ($updated) {
                    create_log('Has been updated a holiday : ' . $data['title']);
                    success($this->lang->line('update_success'));
                    redirect('hrm/holiday');
                } else {
                    error($this->lang->line('update_failed'));
                    redirect('hrm/holiday/edit/' . $this->input->post('id'));
                }
            } else {
                $this->data['holiday'] = $this->holiday->get_single('holidays', array('id' => $this->input->post('id')));
            }
        } else {
            if ($id) {
                $this->data['holiday'] = $this->holiday->get_single('holidays', array('id' => $id));
                if (!$this->data['holiday']) {
                    redirect('hrm/holiday');
                }
            }
        }
        $this->data['edit'] = TRUE;
        $this->layout->title($this->lang->line('edit') . ' ' . $this->global_setting->brand_title);
        $this->layout->view('holiday/edit', $this->data);
    }

    /*     * ***************Function get_single_holiday**********************************
     * @type            : Function
     * @function name   : get_single_holiday
     * @description     : "Load single holiday information" from database                  
     *                    to the user interface   
     * @param           : null
     * @return          : null 
     * ********************************************************** */

    public function get_single_holiday() {
        $id = $this->input->post('id');
        $this->data['holiday'] = $this->holiday->get_single_holiday($id);
        echo $this->load->view('hrm/holiday/view', $this->data);
    }

    /*****************Function _prepare_holiday_validation**********************************
     * @type            : Function
     * @function name   : _prepare_holiday_validation
     * @description     : Process "Holiday" user input data validation
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    private function _prepare_holiday_validation() {
        $this->load->library('form_validation');
        $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
        $this->form_validation->set_rules('title', $this->lang->line('title'), 'trim|required|callback_title');
        $this->form_validation->set_rules('from_date', $this->lang->line('from_date'), 'trim|required');
        $this->form_validation->set_rules('to_date', $this->lang->line('to_date'), 'trim|required');
    }

    /*****************Function title**********************************
     * @type            : Function
     * @function name   : title
     * @description     : Unique check for "Holiday Name" data/value
     *
     * @param           : null
     * @return          : boolean true/false
     * ********************************************************** */
    public function title() {
        if ($this->input->post('id') == '') {
            $holiday = $this->holiday->duplicate_check($this->input->post('from_date'), $this->input->post('title'));
            if ($holiday) {
                $this->form_validation->set_message('title', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else if ($this->input->post('id') != '') {
            $holiday = $this->holiday->duplicate_check($this->input->post('from_date'), $this->input->post('title'), $this->input->post('id'));
            if ($holiday) {
                $this->form_validation->set_message('title', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else {
            return TRUE;
        }
    }

    /*****************Function _get_posted_holiday_data**********************************
     * @type            : Function
     * @function name   : _get_posted_holiday_data
     * @description     : Prepare "Holiday" user input data to save into database
     *
     * @param           : null
     * @return          : $data array(); value
     * ********************************************************** */
    private function _get_posted_holiday_data() {

        $items = array();
        $items[] = 'title';
        $items[] = 'bn_title';
        $items[] = 'remark';
        $data = elements($items, $_POST);

        $data['from_date'] = $this->input->post('from_date') ? date('Y-m-d', strtotime($this->input->post('from_date'))) : null;
        $data['to_date'] = $this->input->post('to_date') ? date('Y-m-d', strtotime($this->input->post('to_date'))) : null;

        $data['is_view_on_web'] = $this->input->post('is_view_on_web') ?? 0;
        $data['status'] = $this->input->post('status') ?? 0;

        if ($this->input->post('id')) {
            $data['modified_at'] = date('Y-m-d H:i:s');
            $data['modified_by'] = logged_in_user_id();
        } else {
            $data['created_at'] = date('Y-m-d H:i:s');
            $data['created_by'] = logged_in_user_id();
        }
        return $data;
    }

    /*****************Function delete**********************************
     * @type            : Function
     * @function name   : delete
     * @description     : delete "Holiday" data from database
     *
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function delete($id = null) {

        check_permission(DELETE);

        if (!is_numeric($id)) {
            error($this->lang->line('unexpected_error'));
            redirect('hrm/holiday');
        }

        $holiday = $this->holiday->get_single('holidays', array('id' => $id));
        if (isset($holiday) && $id > 0) {
            $this->db->trans_start();
            create_log('Has been deleted a holiday : ' . $holiday->title);
            $this->holiday->delete('holidays', array('id' => $id));
            $this->db->trans_complete();
            success($this->lang->line('delete_success'));
        } else {
            error($this->lang->line('delete_failed'));
        }
        redirect($_SERVER['HTTP_REFERER']);
    }
}
