<?php

defined('BASEPATH') or exit('No direct script access allowed');

/* * *****************Testimonial.php**********************************
 * @product name    : CMS Management System
 * @type            : Class
 * @class name      : Testimonial
 * @description     : Manage testimonial frontend website. 
 * @author          : South Bangla IT Zone Team
 * @url             : http://cms.sbitzone.com
 * @support         : rabibd.it@gmail.com
 * @copyright       : South Bangla IT Zone Team
 * ********************************************************** */

class Testimonial extends MY_Controller {

    public $data = array();

    function __construct() {
        parent::__construct();
        $this->load->model('Testimonial_Model', 'testimonial', true);
    }

    /*****************Function index**********************************
     * @type            : Function
     * @function name   : index
     * @description     : Load "Testimonial List" user interface
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function index() {

        check_permission(VIEW);

        $keyword = $this->input->get('keyword');
        $config = array();
        $config = get_pagination(ADMIN_PER_PAGE);
        if ($keyword) {
            $config['base_url'] = site_url('frontend/testimonial/?keyword=' . $keyword);
        } else {
            $config['base_url'] = current_url();
        }
        if ($this->input->get('page')) {
            $sgm = (int) trim($this->input->get('page'));
            $offset = $config['per_page'] * ($sgm - 1);
        } else {
            $offset = 0;
        }

        $config['total_rows'] = $this->testimonial->get_testimonial_list($config['per_page'], $offset, $keyword, $count = TRUE);
        $this->pagination->initialize($config);
        $this->data['testimonials'] = $this->testimonial->get_testimonial_list($config['per_page'], $offset, $keyword, $count = FALSE);
        $this->data['sl_count'] = ($this->input->get('page') == 0 ? 1 : (($this->input->get('page') - 1) * $config["per_page"] + 1));
        $this->data['keyword'] = $keyword;

        $this->data['list'] = TRUE;
        $this->layout->title($this->lang->line('manage_testimonial') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('testimonial/index', $this->data);
    }

    /*****************Function add**********************************
     * @type            : Function
     * @function name   : add
     * @description     : Load "Add new Testimonial" user interface
     *                    and process to store "testimonials" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function add() {

        check_permission(ADD);

        if ($_POST) {
            $this->_prepare_testimonial_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_testimonial_data();
                $insert_id = $this->testimonial->insert('testimonials', $data);
                if ($insert_id) {
                    create_log('Has been created a testimonial : ' . $data['name']);
                    success($this->lang->line('insert_success'));
                    redirect('frontend/testimonial');
                } else {
                    error($this->lang->line('insert_failed'));
                    redirect('frontend/testimonial/add');
                }
            } else {
                $this->data = $_POST;
            }
        }
        $this->data['add'] = TRUE;
        $this->layout->title($this->lang->line('add') . ' ' . $this->lang->line('testimonial') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('testimonial/add', $this->data);
    }

    /*****************Function edit**********************************
     * @type            : Function
     * @function name   : edit
     * @description     : Load Update "Testimonial" user interface
     *                    with populate "Testimonial" value
     *                    and process to update "testimonials" into database
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function edit($id = null) {

        check_permission(EDIT);

        if ($_POST) {
            $this->_prepare_testimonial_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_testimonial_data();
                $updated = $this->testimonial->update('testimonials', $data, array('id' => $this->input->post('id')));
                if ($updated) {
                    create_log('Has been updated a testimonial : ' . $data['name']);
                    success($this->lang->line('update_success'));
                    redirect('frontend/testimonial');
                } else {
                    error($this->lang->line('update_failed'));
                    redirect('testimonial/edit/' . $this->input->post('id'));
                }
            } else {
                $this->data['testimonial'] = $this->testimonial->get_single('testimonials', array('id' => $this->input->post('id')));
            }
        } else {
            if ($id) {
                $this->data['testimonial'] = $this->testimonial->get_single('testimonials', array('id' => $id));
                if (!$this->data['testimonial']) {
                    redirect('frontend/testimonial');
                }
            }
        }
        $this->data['edit'] = TRUE;
        $this->layout->title($this->lang->line('edit') . ' ' . $this->lang->line('testimonial') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('frontend/testimonial/edit', $this->data);
    }

    /*****************Function _prepare_testimonial_validation**********************************
     * @type            : Function
     * @function name   : _prepare_testimonial_validation
     * @description     : Process "testimonials" user input data validation
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    private function _prepare_testimonial_validation() {
        $this->load->library('form_validation');
        $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
        $this->form_validation->set_rules('name', $this->lang->line('name'), 'trim|required');
        $this->form_validation->set_rules('message', $this->lang->line('message'), 'trim|required');
    }

    /*****************Function _get_posted_testimonial_data**********************************
     * @type            : Function
     * @function name   : _get_posted_testimonial_data
     * @description     : Prepare "testimonials" user input data to save into database
     *
     * @param           : null
     * @return          : $data array(); value
     * ********************************************************** */
    private function _get_posted_testimonial_data() {

        $items = array();
        $items[] = 'name';
        $items[] = 'bn_name';
        $items[] = 'company';
        $items[] = 'designation';
        $items[] = 'message';
        $items[] = 'bn_message';
        $items[] = 'rating';
        $data = elements($items, $_POST);

        $data['is_view_on_web'] = $this->input->post('is_view_on_web') ?? 0;
        $data['sort_by'] = $this->input->post('sort_by') ?? 0;


        if ($this->input->post('id')) {
            $data['modified_at'] = date('Y-m-d H:i:s');
            $data['modified_by'] = logged_in_user_id();
        } else {
            $data['status'] = 1;
            $data['created_at'] = date('Y-m-d H:i:s');
            $data['created_by'] = logged_in_user_id();
        }

        if (isset($_FILES['image']['name'])) {
            $data['image'] = $this->_upload_image();
        }

        return $data;
    }


    /*     * ***************Function _upload_image**********************************
     * @type            : Function
     * @function name   : _upload_image
     * @description     : Process to to upload testimonial image in the server
     *                    and return image name                   
     *                       
     * @param           : null
     * @return          : $return_image string value 
     * ********************************************************** */

    private function _upload_image() {

        $prev_image = $this->input->post('prev_image');
        $image = $_FILES['image']['name'];
        $image_type = $_FILES['image']['type'];
        $return_image = '';
        if ($image != "") {
            if (
                $image_type == 'image/jpeg' || $image_type == 'image/pjpeg' ||
                $image_type == 'image/jpg' || $image_type == 'image/png' ||
                $image_type == 'image/x-png' || $image_type == 'image/gif'
            ) {

                $destination = 'assets/uploads/testimonial/';

                $file_type = explode(".", $image);
                $extension = strtolower($file_type[count($file_type) - 1]);
                $image_path = 'testimonial-image-' . time() . '-hms.' . $extension;

                move_uploaded_file($_FILES['image']['tmp_name'], $destination . $image_path);

                // need to unlink previous image
                if ($prev_image != "") {
                    if (file_exists($destination . $prev_image)) {
                        @unlink($destination . $prev_image);
                    }
                }

                $return_image = $image_path;
            }
        } else {
            $return_image = $prev_image;
        }

        return $return_image;
    }

    /*****************Function delete**********************************
     * @type            : Function
     * @function name   : delete
     * @description     : delete "testimonials" data from database
     *
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function delete($id = null) {

        check_permission(DELETE);

        if (!is_numeric($id)) {
            error($this->lang->line('unexpected_error'));
            redirect('frontend/testimonial');
        }

        $testimonial = $this->testimonial->get_single('testimonials', array('id' => $id));

        if ($this->testimonial->delete('testimonials', array('id' => $id))) {
            // delete image
            $destination = 'assets/uploads/';
            if (file_exists($destination . '/testimonial/' . $testimonial->image)) {
                @unlink($destination . '/testimonial/' . $testimonial->image);
            }
            create_log('Has been deleted a testimonial : ' . $testimonial->name);
            success($this->lang->line('delete_success'));
        } else {
            error($this->lang->line('delete_failed'));
        }
        redirect($_SERVER['HTTP_REFERER']);
    }
}
