<?php

defined('BASEPATH') or exit('No direct script access allowed');

/* * *****************Medicine.php**********************************
 * @product name    : Hospital Management System
 * @type            : Class
 * @class name      : Medicine
 * @description     : Manage medicine company.
 * @author          : South Bangla IT Zone Team
 * @url             : http://hms.sbitzone.com
 * @support         : rabibd.it@gmail.com
 * @copyright       : South Bangla IT Zone Team
 * ********************************************************** */

class Medicine extends MY_Controller
{

    public $data = array();

    function __construct()
    {
        parent::__construct();
        $this->load->model('Medicine_Model', 'medicine', true);
    }

    /*****************Function category**********************************
     * @type            : Function
     * @function name   : index
     * @description     : Load "Medicine Category List" user interface
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function category()
    {

        check_permission(VIEW);

        $keyword = trim($this->input->get('keyword'));
        $config = array();
        $config = get_pagination(ADMIN_PER_PAGE);
        if ($keyword) {
            $config['base_url'] = site_url('pharmacy/medicine/category/?search=') . $keyword;
        } else {
            $config['base_url'] = current_url();
        }
        if ($this->input->get('page')) {
            $sgm = (int) trim($this->input->get('page'));
            $offset = $config['per_page'] * ($sgm - 1);
        } else {
            $offset = 0;
        }

        $config['total_rows'] = $this->medicine->get_medicine_category_list($config['per_page'], $offset, $keyword, $count = TRUE);
        $this->pagination->initialize($config);
        $this->data['categories'] = $this->medicine->get_medicine_category_list($config['per_page'], $offset, $keyword, $count = FALSE);
        $this->data['sl_count'] = ($this->input->get('page') == 0 ? 1 : (($this->input->get('page') - 1) * $config["per_page"] + 1));

        $this->data['keyword'] = $keyword;
        $this->data['list'] = TRUE;
        $this->layout->title($this->lang->line('manage_category') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('medicine/category/index', $this->data);
    }

    /*****************Function add_category**********************************
     * @type            : Function
     * @function name   : add_category
     * @description     : Load "Add new medicine category" user interface
     *                    and process to store "medicine_categories" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function add_category()
    {

        check_permission(ADD);

        if ($_POST) {

            $this->load->library('form_validation');
            $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
            $this->form_validation->set_rules('name', $this->lang->line('name'), 'trim|required|callback_category_name');

            if ($this->form_validation->run() === TRUE) {
                $data['name'] = $this->input->post('name');
                $data['note'] = $this->input->post('note');
                $data['sort_by'] = $this->input->post('sort_by');
                $data['status'] = $this->input->post('status') ? $this->input->post('status') : 0;
                $data['created_at'] = date('Y-m-d H:i:s');
                $data['created_by'] = logged_in_user_id();

                $insert_id = $this->medicine->insert('medicine_categories', $data);
                if ($insert_id) {
                    create_log('Has been created a category : ' . $data['name']);
                    success($this->lang->line('insert_success'));
                    redirect('pharmacy/medicine/category');
                } else {
                    error($this->lang->line('insert_failed'));
                    redirect('pharmacy/medicine/add_category');
                }
            } else {
                $this->data = $_POST;
            }
        }

        $this->data['add'] = TRUE;
        $this->layout->title($this->lang->line('add') . ' ' . $this->lang->line('category') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('pharmacy/medicine/category/add', $this->data);
    }

    /*****************Function edit_category**********************************
     * @type            : Function
     * @function name   : edit_category
     * @description     : Load "Edit new medicine category" user interface
     *                    and process to store "medicine_categories" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function edit_category($id = NULL)
    {

        check_permission(EDIT);

        if ($_POST) {

            $this->load->library('form_validation');
            $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
            $this->form_validation->set_rules('name', $this->lang->line('name'), 'trim|required|callback_category_name');

            if ($this->form_validation->run() === TRUE) {
                $data['name'] = $this->input->post('name');
                $data['note'] = $this->input->post('note');
                $data['sort_by'] = $this->input->post('sort_by');
                $data['status'] = $this->input->post('status') ? $this->input->post('status') : 0;
                $data['modified_at'] = date('Y-m-d H:i:s');
                $data['modified_by'] = logged_in_user_id();

                $updated = $this->medicine->update('medicine_categories', $data, array('id' => $this->input->post('id')));
                if ($updated) {
                    create_log('Has been updated a category : ' . $data['name']);
                    success($this->lang->line('update_success'));
                    redirect('pharmacy/medicine/category');
                } else {
                    error($this->lang->line('update_failed'));
                    redirect('pharmacy/medicine/edit_category/' . $this->input->post('id'));
                }
            } else {
                $this->data['category'] = $this->medicine->get_single('medicine_categories', array('id' => $this->input->post('id')));
            }
        } else {
            if ($id) {
                $this->data['category'] = $this->medicine->get_single('medicine_categories', array('id' => $id));
                if (!$this->data['category']) {
                    redirect('pharmacy/medicine/category');
                }
            }
        }

        $this->data['edit'] = TRUE;
        $this->layout->title($this->lang->line('edit') . ' ' . $this->lang->line('category') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('pharmacy/medicine/category/edit', $this->data);
    }

    /*****************Function category_name**********************************
     * @type            : Function
     * @function name   : category_name
     * @description     : Unique check for "Category Name" data/value
     *
     * @param           : null
     * @return          : boolean true/false
     * ********************************************************** */
    public function category_name()
    {
        if ($this->input->post('id') == '') {
            $category = $this->medicine->duplicate_check_category_name($this->input->post('name'));
            if ($category) {
                $this->form_validation->set_message('category_name', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else if ($this->input->post('id') != '') {
            $category = $this->medicine->duplicate_check_category_name($this->input->post('name'), $this->input->post('id'));
            if ($category) {
                $this->form_validation->set_message('category_name', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else {
            return TRUE;
        }
    }

    /*****************Function delete_category**********************************
     * @type            : Function
     * @function name   : delete_category
     * @description     : delete "category" data from database
     *
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function delete_category($id = null)
    {

        check_permission(DELETE);

        if (!is_numeric($id)) {
            error($this->lang->line('unexpected_error'));
            redirect('pharmacy/medicine/category');
        }

        $category = $this->medicine->get_single('medicine_categories', array('id' => $id));

        if ($this->medicine->delete('medicine_categories', array('id' => $id))) {
            create_log('Has been deleted a type : ' . $category->name);
            success($this->lang->line('delete_success'));
        } else {
            error($this->lang->line('delete_failed'));
        }
        redirect($_SERVER['HTTP_REFERER']);
    }

    /*****************Function dosage**********************************
     * @type            : Function
     * @function name   : index
     * @description     : Load "Medicine Dosages List" user interface
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function dosage()
    {

        check_permission(VIEW);

        $category_id = trim($this->input->get('category_id'));
        $keyword = trim($this->input->get('keyword'));
        $config = array();
        $config = get_pagination(ADMIN_PER_PAGE);
        if ($category_id || $keyword) {
            $config['base_url'] = site_url('pharmacy/medicine/dosage/?category_id=' . $category_id . '&search=' . $keyword);
        } else {
            $config['base_url'] = current_url();
        }
        if ($this->input->get('page')) {
            $sgm = (int) trim($this->input->get('page'));
            $offset = $config['per_page'] * ($sgm - 1);
        } else {
            $offset = 0;
        }

        $config['total_rows'] = $this->medicine->get_medicine_dosage_list($config['per_page'], $offset, $category_id, $keyword, $count = TRUE);
        $this->pagination->initialize($config);
        $this->data['dosages'] = $this->medicine->get_medicine_dosage_list($config['per_page'], $offset, $category_id, $keyword, $count = FALSE);
        $this->data['sl_count'] = ($this->input->get('page') == 0 ? 1 : (($this->input->get('page') - 1) * $config["per_page"] + 1));

        $this->data['categories'] = $this->medicine->get_list('medicine_categories', array('status' => 1), '', '', '', 'id', 'ASC');
        $this->data['category_id'] = $category_id;
        $this->data['keyword'] = $keyword;
        $this->data['list'] = TRUE;
        $this->layout->title($this->lang->line('manage_dosage') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('medicine/dosage/index', $this->data);
    }

    /*****************Function add_dosage**********************************
     * @type            : Function
     * @function name   : add_dosage
     * @description     : Load "Add new medicine dosage" user interface
     *                    and process to store "medicine_dosages" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function add_dosage()
    {

        check_permission(ADD);

        if ($_POST) {

            $this->load->library('form_validation');
            $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
            $this->form_validation->set_rules('category_id', $this->lang->line('category'), 'trim');
            $this->form_validation->set_rules('name', $this->lang->line('dosage') . ' ' . $this->lang->line('name'), 'trim|required|callback_dosage_name');
            $this->form_validation->set_rules('short_name', $this->lang->line('short_name'), 'trim');

            if ($this->form_validation->run() === TRUE) {
                $data['category_id'] = $this->input->post('category_id');
                $data['name'] = $this->input->post('name');
                $data['short_name'] = $this->input->post('short_name');
                $data['note'] = $this->input->post('note');
                $data['sort_by'] = $this->input->post('sort_by') ? $this->input->post('sort_by') : 0;
                $data['status'] = $this->input->post('status') ? $this->input->post('status') : 0;
                $data['created_at'] = date('Y-m-d H:i:s');
                $data['created_by'] = logged_in_user_id();

                $insert_id = $this->medicine->insert('medicine_dosages', $data);
                if ($insert_id) {
                    create_log('Has been created a dosage : ' . $data['name']);
                    success($this->lang->line('insert_success'));
                    redirect('pharmacy/medicine/dosage');
                } else {
                    error($this->lang->line('insert_failed'));
                    redirect('pharmacy/medicine/add_dosage');
                }
            } else {
                $this->data = $_POST;
            }
        }

        $this->data['categories'] = $this->medicine->get_list('medicine_categories', array('status' => 1), '', '', '', 'id', 'ASC');
        $this->data['add'] = TRUE;
        $this->layout->title($this->lang->line('add') . ' ' . $this->lang->line('dosage') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('pharmacy/medicine/dosage/add', $this->data);
    }

    /*****************Function edit_dosage**********************************
     * @type            : Function
     * @function name   : edit_dosage
     * @description     : Load "Edit new medicine dosage" user interface
     *                    and process to store "medicine_dosage" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function edit_dosage($id = NULL)
    {

        check_permission(EDIT);

        if ($_POST) {

            $this->load->library('form_validation');
            $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
            $this->form_validation->set_rules('category_id', $this->lang->line('category'), 'trim');
            $this->form_validation->set_rules('name', $this->lang->line('dosage') . ' ' . $this->lang->line('name'), 'trim|required|callback_dosage_name');
            $this->form_validation->set_rules('short_name', $this->lang->line('short_name'), 'trim');

            if ($this->form_validation->run() === TRUE) {
                $data['category_id'] = $this->input->post('category_id');
                $data['name'] = $this->input->post('name');
                $data['short_name'] = $this->input->post('short_name');
                $data['note'] = $this->input->post('note');
                $data['sort_by'] = $this->input->post('sort_by') ? $this->input->post('sort_by') : 0;
                $data['status'] = $this->input->post('status') ? $this->input->post('status') : 0;
                $data['modified_at'] = date('Y-m-d H:i:s');
                $data['modified_by'] = logged_in_user_id();

                $updated = $this->medicine->update('medicine_dosages', $data, array('id' => $this->input->post('id')));
                if ($updated) {
                    create_log('Has been updated a dosage : ' . $data['name']);
                    success($this->lang->line('update_success'));
                    redirect('pharmacy/medicine/dosage');
                } else {
                    error($this->lang->line('update_failed'));
                    redirect('pharmacy/medicine/edit_dosage/' . $this->input->post('id'));
                }
            } else {
                $this->data['dosage'] = $this->medicine->get_single('medicine_dosages', array('id' => $this->input->post('id')));
            }
        } else {
            if ($id) {
                $this->data['dosage'] = $this->medicine->get_single('medicine_dosages', array('id' => $id));
                if (!$this->data['dosage']) {
                    redirect('pharmacy/medicine/dosage');
                }
            }
        }

        $this->data['categories'] = $this->medicine->get_list('medicine_categories', array('status' => 1), '', '', '', 'id', 'ASC');
        $this->data['edit'] = TRUE;
        $this->layout->title($this->lang->line('edit') . ' ' . $this->lang->line('dosage') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('pharmacy/medicine/dosage/edit', $this->data);
    }

    /*****************Function dosage_name**********************************
     * @type            : Function
     * @function name   : dosage_name
     * @description     : Unique check for "Dosage Name" data/value
     *
     * @param           : null
     * @return          : boolean true/false
     * ********************************************************** */
    public function dosage_name()
    {
        if ($this->input->post('id') == '') {
            $dosage = $this->medicine->duplicate_check_dosage_name($this->input->post('name'));
            if ($dosage) {
                $this->form_validation->set_message('name', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else if ($this->input->post('id') != '') {
            $dosage = $this->medicine->duplicate_check_dosage_name($this->input->post('name'), $this->input->post('id'));
            if ($dosage) {
                $this->form_validation->set_message('name', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else {
            return TRUE;
        }
    }

    /*****************Function delete_dosage**********************************
     * @type            : Function
     * @function name   : delete_dosage
     * @description     : delete "dosage" data from database
     *
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function delete_dosage($id = null)
    {

        check_permission(DELETE);

        if (!is_numeric($id)) {
            error($this->lang->line('unexpected_error'));
            redirect('pharmacy/medicine/dosage');
        }

        $dosage = $this->medicine->get_single('medicine_dosages', array('id' => $id));

        if ($this->medicine->delete('medicine_dosages', array('id' => $id))) {
            create_log('Has been deleted a dosage : ' . $dosage->name);
            success($this->lang->line('delete_success'));
        } else {
            error($this->lang->line('delete_failed'));
        }
        redirect($_SERVER['HTTP_REFERER']);
    }


    /*****************Function type**********************************
     * @type            : Function
     * @function name   : index
     * @description     : Load "Medicine Type List" user interface
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function type()
    {

        check_permission(VIEW);

        $keyword = trim($this->input->get('keyword'));
        $config = array();
        $config = get_pagination(ADMIN_PER_PAGE);
        if ($keyword) {
            $config['base_url'] = site_url('pharmacy/medicine/type/?search=') . $keyword;
        } else {
            $config['base_url'] = current_url();
        }
        if ($this->input->get('page')) {
            $sgm = (int) trim($this->input->get('page'));
            $offset = $config['per_page'] * ($sgm - 1);
        } else {
            $offset = 0;
        }

        $config['total_rows'] = $this->medicine->get_medicine_type_list($config['per_page'], $offset, $keyword, $count = TRUE);
        $this->pagination->initialize($config);
        $this->data['types'] = $this->medicine->get_medicine_type_list($config['per_page'], $offset, $keyword, $count = FALSE);
        $this->data['sl_count'] = ($this->input->get('page') == 0 ? 1 : (($this->input->get('page') - 1) * $config["per_page"] + 1));

        $this->data['keyword'] = $keyword;
        $this->data['list'] = TRUE;
        $this->layout->title($this->lang->line('manage_type') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('medicine/type/index', $this->data);
    }

    /*****************Function add_type**********************************
     * @type            : Function
     * @function name   : add_type
     * @description     : Load "Add new medicine type" user interface
     *                    and process to store "medicine_types" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function add_type()
    {

        check_permission(ADD);

        if ($_POST) {

            $this->load->library('form_validation');
            $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
            $this->form_validation->set_rules('name', $this->lang->line('name'), 'trim|required|callback_type_name');

            if ($this->form_validation->run() === TRUE) {
                $data['name'] = $this->input->post('name');
                $data['note'] = $this->input->post('note');
                $data['sort_by'] = $this->input->post('sort_by');
                $data['status'] = $this->input->post('status') ? $this->input->post('status') : 0;
                $data['created_at'] = date('Y-m-d H:i:s');
                $data['created_by'] = logged_in_user_id();

                $insert_id = $this->medicine->insert('medicine_types', $data);
                if ($insert_id) {
                    create_log('Has been created a type : ' . $data['name']);
                    success($this->lang->line('insert_success'));
                    redirect('pharmacy/medicine/type');
                } else {
                    error($this->lang->line('insert_failed'));
                    redirect('pharmacy/medicine/add_type');
                }
            } else {
                $this->data = $_POST;
            }
        }

        $this->data['add'] = TRUE;
        $this->layout->title($this->lang->line('add') . ' ' . $this->lang->line('type') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('pharmacy/medicine/type/add', $this->data);
    }

    /*****************Function edit_type**********************************
     * @type            : Function
     * @function name   : edit_type
     * @description     : Load "Edit new medicine type" user interface
     *                    and process to store "medicine_types" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function edit_type($id = NULL)
    {

        check_permission(EDIT);

        if ($_POST) {

            $this->load->library('form_validation');
            $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
            $this->form_validation->set_rules('name', $this->lang->line('name'), 'trim|required|callback_type_name');

            if ($this->form_validation->run() === TRUE) {
                $data['name'] = $this->input->post('name');
                $data['note'] = $this->input->post('note');
                $data['sort_by'] = $this->input->post('sort_by');
                $data['status'] = $this->input->post('status') ? $this->input->post('status') : 0;
                $data['modified_at'] = date('Y-m-d H:i:s');
                $data['modified_by'] = logged_in_user_id();

                $updated = $this->medicine->update('medicine_types', $data, array('id' => $this->input->post('id')));
                if ($updated) {
                    create_log('Has been updated a type : ' . $data['name']);
                    success($this->lang->line('update_success'));
                    redirect('pharmacy/medicine/type');
                } else {
                    error($this->lang->line('update_failed'));
                    redirect('pharmacy/medicine/edit_type/' . $this->input->post('id'));
                }
            } else {
                $this->data['type'] = $this->medicine->get_single('medicine_types', array('id' => $this->input->post('id')));
            }
        } else {
            if ($id) {
                $this->data['type'] = $this->medicine->get_single('medicine_types', array('id' => $id));
                if (!$this->data['type']) {
                    redirect('pharmacy/medicine/type');
                }
            }
        }

        $this->data['edit'] = TRUE;
        $this->layout->title($this->lang->line('edit') . ' ' . $this->lang->line('type') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('pharmacy/medicine/type/edit', $this->data);
    }

    /*****************Function type_name**********************************
     * @type            : Function
     * @function name   : type_name
     * @description     : Unique check for "Type Name" data/value
     *
     * @param           : null
     * @return          : boolean true/false
     * ********************************************************** */
    public function type_name()
    {
        if ($this->input->post('id') == '') {
            $type = $this->medicine->duplicate_check_type_name($this->input->post('name'));
            if ($type) {
                $this->form_validation->set_message('type_name', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else if ($this->input->post('id') != '') {
            $type = $this->medicine->duplicate_check_type_name($this->input->post('name'), $this->input->post('id'));
            if ($type) {
                $this->form_validation->set_message('type_name', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else {
            return TRUE;
        }
    }

    /*****************Function delete_type**********************************
     * @type            : Function
     * @function name   : delete_type
     * @description     : delete "type" data from database
     *
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function delete_type($id = null)
    {

        check_permission(DELETE);

        if (!is_numeric($id)) {
            error($this->lang->line('unexpected_error'));
            redirect('pharmacy/medicine/type');
        }

        $type = $this->medicine->get_single('medicine_types', array('id' => $id));

        if ($this->medicine->delete('medicine_types', array('id' => $id))) {
            create_log('Has been deleted a type : ' . $type->name);
            success($this->lang->line('delete_success'));
        } else {
            error($this->lang->line('delete_failed'));
        }
        redirect($_SERVER['HTTP_REFERER']);
    }


    /*****************Function generic**********************************
     * @type            : Function
     * @function name   : index
     * @description     : Load "Medicine Generic List" user interface
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function generic()
    {

        check_permission(VIEW);

        $keyword = trim($this->input->get('keyword'));
        $config = array();
        $config = get_pagination(ADMIN_PER_PAGE);
        if ($keyword) {
            $config['base_url'] = site_url('pharmacy/medicine/generic/?search=') . $keyword;
        } else {
            $config['base_url'] = current_url();
        }
        if ($this->input->get('page')) {
            $sgm = (int) trim($this->input->get('page'));
            $offset = $config['per_page'] * ($sgm - 1);
        } else {
            $offset = 0;
        }

        $config['total_rows'] = $this->medicine->get_medicine_generic_list($config['per_page'], $offset, $keyword, $count = TRUE);
        $this->pagination->initialize($config);
        $this->data['generics'] = $this->medicine->get_medicine_generic_list($config['per_page'], $offset, $keyword, $count = FALSE);
        $this->data['sl_count'] = ($this->input->get('page') == 0 ? 1 : (($this->input->get('page') - 1) * $config["per_page"] + 1));

        $this->data['keyword'] = $keyword;
        $this->data['list'] = TRUE;
        $this->layout->title($this->lang->line('manage_generic') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('medicine/generic/index', $this->data);
    }

    /*****************Function add_generic**********************************
     * @type            : Function
     * @function name   : add_generic
     * @description     : Load "Add new medicine generic" user interface
     *                    and process to store "medicine_generics" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function add_generic()
    {

        check_permission(ADD);

        if ($_POST) {

            $this->load->library('form_validation');
            $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
            $this->form_validation->set_rules('name', $this->lang->line('name'), 'trim|required|callback_generic_name');

            if ($this->form_validation->run() === TRUE) {
                $data['type_id'] = $this->input->post('type_id');
                $data['name'] = $this->input->post('name');
                $data['note'] = $this->input->post('note');
                $data['sort_by'] = $this->input->post('sort_by');
                $data['status'] = $this->input->post('status') ? $this->input->post('status') : 0;
                $data['created_at'] = date('Y-m-d H:i:s');
                $data['created_by'] = logged_in_user_id();

                $insert_id = $this->medicine->insert('medicine_generics', $data);
                if ($insert_id) {
                    create_log('Has been created a generic : ' . $data['name']);
                    success($this->lang->line('insert_success'));
                    redirect('pharmacy/medicine/generic');
                } else {
                    error($this->lang->line('insert_failed'));
                    redirect('pharmacy/medicine/add_generic');
                }
            } else {
                $this->data = $_POST;
            }
        }

        $this->data['types'] = $this->medicine->get_list('medicine_types', array('status' => 1), '', '', '', 'id', 'ASC');
        $this->data['add'] = TRUE;
        $this->layout->title($this->lang->line('add') . ' ' . $this->lang->line('generic') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('pharmacy/medicine/generic/add', $this->data);
    }

    /*****************Function edit_generic**********************************
     * @type            : Function
     * @function name   : edit_generic
     * @description     : Load "Edit new medicine generic" user interface
     *                    and process to store "medicine_generics" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function edit_generic($id = NULL)
    {

        check_permission(EDIT);

        if ($_POST) {

            $this->load->library('form_validation');
            $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
            $this->form_validation->set_rules('name', $this->lang->line('name'), 'trim|required|callback_generic_name');

            if ($this->form_validation->run() === TRUE) {
                $data['type_id'] = $this->input->post('type_id');
                $data['name'] = $this->input->post('name');
                $data['note'] = $this->input->post('note');
                $data['sort_by'] = $this->input->post('sort_by');
                $data['status'] = $this->input->post('status') ? $this->input->post('status') : 0;
                $data['modified_at'] = date('Y-m-d H:i:s');
                $data['modified_by'] = logged_in_user_id();

                $updated = $this->medicine->update('medicine_generics', $data, array('id' => $this->input->post('id')));
                if ($updated) {
                    create_log('Has been updated a generic : ' . $data['name']);
                    success($this->lang->line('update_success'));
                    redirect('pharmacy/medicine/generic');
                } else {
                    error($this->lang->line('update_failed'));
                    redirect('pharmacy/medicine/edit_generic/' . $this->input->post('id'));
                }
            } else {
                $this->data['generic'] = $this->medicine->get_single('medicine_generics', array('id' => $this->input->post('id')));
            }
        } else {
            if ($id) {
                $this->data['generic'] = $this->medicine->get_single('medicine_generics', array('id' => $id));
                if (!$this->data['generic']) {
                    redirect('pharmacy/medicine/generic');
                }
            }
        }

        $this->data['types'] = $this->medicine->get_list('medicine_types', array('status' => 1), '', '', '', 'id', 'ASC');
        $this->data['edit'] = TRUE;
        $this->layout->title($this->lang->line('edit') . ' ' . $this->lang->line('generic') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('pharmacy/medicine/generic/edit', $this->data);
    }

    /*****************Function generic_name**********************************
     * @type            : Function
     * @function name   : generic_name
     * @description     : Unique check for "Generic Name" data/value
     *
     * @param           : null
     * @return          : boolean true/false
     * ********************************************************** */
    public function generic_name()
    {
        if ($this->input->post('id') == '') {
            $generic = $this->medicine->duplicate_check_generic_name($this->input->post('name'));
            if ($generic) {
                $this->form_validation->set_message('generic_name', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else if ($this->input->post('id') != '') {
            $generic = $this->medicine->duplicate_check_generic_name($this->input->post('name'), $this->input->post('id'));
            if ($generic) {
                $this->form_validation->set_message('generic_name', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else {
            return TRUE;
        }
    }

    /*****************Function delete_generic**********************************
     * @type            : Function
     * @function name   : delete_generic
     * @description     : delete "generic" data from database
     *
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function delete_generic($id = null)
    {

        check_permission(DELETE);

        if (!is_numeric($id)) {
            error($this->lang->line('unexpected_error'));
            redirect('pharmacy/medicine/generic');
        }

        $generic = $this->medicine->get_single('medicine_generics', array('id' => $id));

        if ($this->medicine->delete('medicine_generics', array('id' => $id))) {
            create_log('Has been deleted a generic : ' . $generic->name);
            success($this->lang->line('delete_success'));
        } else {
            error($this->lang->line('delete_failed'));
        }
        redirect($_SERVER['HTTP_REFERER']);
    }


    /*****************Function unit**********************************
     * @type            : Function
     * @function name   : index
     * @description     : Load "Medicine Unit List" user interface
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function unit()
    {

        check_permission(VIEW);

        $keyword = trim($this->input->get('keyword'));
        $config = array();
        $config = get_pagination(ADMIN_PER_PAGE);
        if ($keyword) {
            $config['base_url'] = site_url('pharmacy/medicine/unit/?search=') . $keyword;
        } else {
            $config['base_url'] = current_url();
        }
        if ($this->input->get('page')) {
            $sgm = (int) trim($this->input->get('page'));
            $offset = $config['per_page'] * ($sgm - 1);
        } else {
            $offset = 0;
        }

        $config['total_rows'] = $this->medicine->get_medicine_unit_list($config['per_page'], $offset, $keyword, $count = TRUE);
        $this->pagination->initialize($config);
        $this->data['units'] = $this->medicine->get_medicine_unit_list($config['per_page'], $offset, $keyword, $count = FALSE);
        $this->data['sl_count'] = ($this->input->get('page') == 0 ? 1 : (($this->input->get('page') - 1) * $config["per_page"] + 1));

        $this->data['keyword'] = $keyword;
        $this->data['list'] = TRUE;
        $this->layout->title($this->lang->line('manage_unit') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('medicine/unit/index', $this->data);
    }

    /*****************Function add_unit**********************************
     * @type            : Function
     * @function name   : add_unit
     * @description     : Load "Add new medicine unit" user interface
     *                    and process to store "medicine_units" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function add_unit()
    {

        check_permission(ADD);

        if ($_POST) {

            $this->load->library('form_validation');
            $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
            $this->form_validation->set_rules('name', $this->lang->line('name'), 'trim|required|callback_unit_name');

            if ($this->form_validation->run() === TRUE) {
                $data['name'] = $this->input->post('name');
                $data['note'] = $this->input->post('note');
                $data['sort_by'] = $this->input->post('sort_by');
                $data['status'] = $this->input->post('status') ? $this->input->post('status') : 0;
                $data['created_at'] = date('Y-m-d H:i:s');
                $data['created_by'] = logged_in_user_id();

                $insert_id = $this->medicine->insert('medicine_units', $data);
                if ($insert_id) {
                    create_log('Has been created a unit : ' . $data['name']);
                    success($this->lang->line('insert_success'));
                    redirect('pharmacy/medicine/unit');
                } else {
                    error($this->lang->line('insert_failed'));
                    redirect('pharmacy/medicine/add_unit');
                }
            } else {
                $this->data = $_POST;
            }
        }

        $this->data['add'] = TRUE;
        $this->layout->title($this->lang->line('add') . ' ' . $this->lang->line('unit') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('pharmacy/medicine/unit/add', $this->data);
    }

    /*****************Function edit_unit**********************************
     * @type            : Function
     * @function name   : edit_unit
     * @description     : Load "Edit new medicine unit" user interface
     *                    and process to store "medicine_units" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function edit_unit($id = NULL)
    {

        check_permission(EDIT);

        if ($_POST) {

            $this->load->library('form_validation');
            $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
            $this->form_validation->set_rules('name', $this->lang->line('name'), 'trim|required|callback_unit_name');

            if ($this->form_validation->run() === TRUE) {
                $data['name'] = $this->input->post('name');
                $data['note'] = $this->input->post('note');
                $data['sort_by'] = $this->input->post('sort_by');
                $data['status'] = $this->input->post('status') ? $this->input->post('status') : 0;
                $data['modified_at'] = date('Y-m-d H:i:s');
                $data['modified_by'] = logged_in_user_id();

                $updated = $this->medicine->update('medicine_units', $data, array('id' => $this->input->post('id')));
                if ($updated) {
                    create_log('Has been updated a unit : ' . $data['name']);
                    success($this->lang->line('update_success'));
                    redirect('pharmacy/medicine/unit');
                } else {
                    error($this->lang->line('update_failed'));
                    redirect('pharmacy/medicine/edit_unit/' . $this->input->post('id'));
                }
            } else {
                $this->data['unit'] = $this->medicine->get_single('medicine_units', array('id' => $this->input->post('id')));
            }
        } else {
            if ($id) {
                $this->data['unit'] = $this->medicine->get_single('medicine_units', array('id' => $id));
                if (!$this->data['unit']) {
                    redirect('pharmacy/medicine/unit');
                }
            }
        }

        $this->data['edit'] = TRUE;
        $this->layout->title($this->lang->line('edit') . ' ' . $this->lang->line('unit') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('pharmacy/medicine/unit/edit', $this->data);
    }

    /*****************Function unit_name**********************************
     * @type            : Function
     * @function name   : unit_name
     * @description     : Unique check for "Generic Name" data/value
     *
     * @param           : null
     * @return          : boolean true/false
     * ********************************************************** */
    public function unit_name()
    {
        if ($this->input->post('id') == '') {
            $unit = $this->medicine->duplicate_check_unit_name($this->input->post('name'));
            if ($unit) {
                $this->form_validation->set_message('unit_name', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else if ($this->input->post('id') != '') {
            $unit = $this->medicine->duplicate_check_unit_name($this->input->post('name'), $this->input->post('id'));
            if ($unit) {
                $this->form_validation->set_message('unit_name', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else {
            return TRUE;
        }
    }

    /*****************Function delete_unit**********************************
     * @type            : Function
     * @function name   : delete_unit
     * @description     : delete "unit" data from database
     *
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function delete_unit($id = null)
    {

        check_permission(DELETE);

        if (!is_numeric($id)) {
            error($this->lang->line('unexpected_error'));
            redirect('pharmacy/medicine/unit');
        }

        $unit = $this->medicine->get_single('medicine_units', array('id' => $id));

        if ($this->medicine->delete('medicine_units', array('id' => $id))) {
            create_log('Has been deleted a unit : ' . $unit->name);
            success($this->lang->line('delete_success'));
        } else {
            error($this->lang->line('delete_failed'));
        }
        redirect($_SERVER['HTTP_REFERER']);
    }


    /*****************Function index**********************************
     * @type            : Function
     * @function name   : index
     * @description     : Load "Medicine List" user interface
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function index()
    {

        check_permission(VIEW);

        $keyword = trim($this->input->get('keyword'));
        $type_id = $this->input->get('type_id');
        $category_id = $this->input->get('category_id');
        $manufacturer_id = $this->input->get('manufacturer_id');
        $generic_id = $this->input->get('generic_id');
        $unit_id = $this->input->get('unit_id');

        $config = array();
        $config = get_pagination(ADMIN_PER_PAGE);
        if ($keyword || $type_id || $category_id || $manufacturer_id || $generic_id || $unit_id) {
            $config['base_url'] = site_url('pharmacy/medicine/?search=' . $keyword . '&type_id=' . $type_id . '&category_id=' . $category_id . '&manufacturer_id=' . $manufacturer_id . '&generic_id=' . $generic_id . '&unit_id=' . $unit_id);
        } else {
            $config['base_url'] = current_url();
        }
        if ($this->input->get('page')) {
            $sgm = (int) trim($this->input->get('page'));
            $offset = $config['per_page'] * ($sgm - 1);
        } else {
            $offset = 0;
        }

        $config['total_rows'] = $this->medicine->get_medicine_list($config['per_page'], $offset, $keyword, $type_id, $category_id, $manufacturer_id, $generic_id, $unit_id, $count = TRUE);
        $this->pagination->initialize($config);
        $this->data['medicines'] = $this->medicine->get_medicine_list($config['per_page'], $offset, $keyword, $type_id, $category_id, $manufacturer_id, $generic_id, $unit_id, $count = FALSE);
        $this->data['sl_count'] = ($this->input->get('page') == 0 ? 1 : (($this->input->get('page') - 1) * $config["per_page"] + 1));


        $this->data['keyword'] = $keyword;
        $this->data['type_id'] = $type_id;
        $this->data['category_id'] = $category_id;
        $this->data['manufacturer_id'] = $manufacturer_id;
        $this->data['generic_id'] = $generic_id;
        $this->data['unit_id'] = $unit_id;

        $this->data['types'] = $this->medicine->get_list('medicine_types', array('status' => 1), '', '', '', 'id', 'ASC');
        $this->data['categories'] = $this->medicine->get_list('medicine_categories', array('status' => 1), '', '', '', 'id', 'ASC');
        $this->data['manufacturers'] = $this->medicine->get_list('manufacturers', array('status' => 1), '', '', '', 'id', 'ASC');
        $this->data['generics'] = $this->medicine->get_list('medicine_generics', array('status' => 1), '', '', '', 'id', 'ASC');
        $this->data['units'] = $this->medicine->get_list('medicine_units', array('status' => 1), '', '', '', 'id', 'ASC');

        $this->data['list'] = TRUE;
        $this->layout->title($this->lang->line('manage_medicine') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('pharmacy/medicine/index', $this->data);
    }

    /*****************Function add**********************************
     * @type            : Function
     * @function name   : add
     * @description     : Load "Add new medicine" user interface
     *                    and process to store "medicines" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function add()
    {

        check_permission(ADD);

        if ($_POST) {

            $this->load->library('form_validation');
            $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
            $this->form_validation->set_rules('type_id', $this->lang->line('type'), 'trim');
            $this->form_validation->set_rules('category_id', $this->lang->line('category'), 'trim');
            $this->form_validation->set_rules('manufacturer_id', $this->lang->line('manufacturer') . ' ' . $this->lang->line('name'), 'trim');
            $this->form_validation->set_rules('generic_id', $this->lang->line('generic') . ' ' . $this->lang->line('name'), 'trim');
            $this->form_validation->set_rules('unit_id', $this->lang->line('unit'), 'trim');
            $this->form_validation->set_rules('medicine_name', $this->lang->line('medicine') . ' ' . $this->lang->line('name'), 'trim|required|callback_medicine_name');
            $this->form_validation->set_rules('mrp', $this->lang->line('mrp'), 'trim');

            if ($this->form_validation->run() === TRUE) {
                $data['code'] = get_investagation_custom_code('medicines', 'M');
                $data['barcode'] = get_custom_invoice('medicines', '8');
                $data['type_id'] = $this->input->post('type_id');
                $data['category_id'] = $this->input->post('category_id');
                $data['manufacturer_id'] = $this->input->post('manufacturer_id');
                $data['generic_id'] = $this->input->post('generic_id');
                $data['unit_id'] = $this->input->post('unit_id');
                $data['strength'] = $this->input->post('strength');
                $data['medicine_name'] = $this->input->post('medicine_name');
                $data['rack_no'] = $this->input->post('rack_no');
                $data['mrp'] = $this->input->post('mrp');
                $data['purchase_price'] = $this->input->post('purchase_price');
                $data['sell_price'] = $this->input->post('sell_price');
                $data['whole_sale_price'] = $this->input->post('whole_sale_price');
                $data['minimum_stock'] = $this->input->post('minimum_stock');
                $data['note'] = $this->input->post('note');
                $data['status'] = $this->input->post('status') ? $this->input->post('status') : 0;
                $data['created_at'] = date('Y-m-d H:i:s');
                $data['created_by'] = logged_in_user_id();

                if ($_FILES['photo']['name']) {
                    $data['photo'] = $this->_medicine_photo();
                }

                $insert_id = $this->medicine->insert('medicines', $data);
                if ($insert_id) {
                    create_log('Has been created a medicine : ' . $data['medicine_name']);
                    success($this->lang->line('insert_success'));
                    redirect('pharmacy/medicine');
                } else {
                    error($this->lang->line('insert_failed'));
                    redirect('pharmacy/medicine/add');
                }
            } else {
                $this->data = $_POST;
            }
        }

        $this->data['types'] = $this->medicine->get_list('medicine_types', array('status' => 1), '', '', '', 'id', 'ASC');
        $this->data['categories'] = $this->medicine->get_list('medicine_categories', array('status' => 1), '', '', '', 'id', 'ASC');
        $this->data['manufacturers'] = $this->medicine->get_list('manufacturers', array('status' => 1), '', '', '', 'id', 'ASC');
        $this->data['generics'] = $this->medicine->get_list('medicine_generics', array('status' => 1), '', '', '', 'id', 'DESC');
        $this->data['units'] = $this->medicine->get_list('medicine_units', array('status' => 1), '', '', '', 'id', 'ASC');

        $this->data['add'] = TRUE;
        $this->layout->title($this->lang->line('add') . ' ' . $this->lang->line('medicine') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('pharmacy/medicine/add', $this->data);
    }

    /*****************Function edit**********************************
     * @type            : Function
     * @function name   : edit
     * @description     : Load "Edit new medicine" user interface
     *                    and process to store "medicines" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function edit($id = NULL)
    {

        check_permission(EDIT);

        if ($_POST) {

            $this->load->library('form_validation');
            $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
            $this->form_validation->set_rules('type_id', $this->lang->line('type'), 'trim');
            $this->form_validation->set_rules('category_id', $this->lang->line('category'), 'trim');
            $this->form_validation->set_rules('manufacturer_id', $this->lang->line('manufacturer') . ' ' . $this->lang->line('name'), 'trim');
            $this->form_validation->set_rules('generic_id', $this->lang->line('generic') . ' ' . $this->lang->line('name'), 'trim');
            $this->form_validation->set_rules('unit_id', $this->lang->line('unit'), 'trim');
            $this->form_validation->set_rules('medicine_name', $this->lang->line('medicine') . ' ' . $this->lang->line('name'), 'trim|required|callback_medicine_name');
            $this->form_validation->set_rules('mrp', $this->lang->line('mrp'), 'trim');

            if ($this->form_validation->run() === TRUE) {
                $data['type_id'] = $this->input->post('type_id');
                $data['category_id'] = $this->input->post('category_id');
                $data['manufacturer_id'] = $this->input->post('manufacturer_id');
                $data['generic_id'] = $this->input->post('generic_id');
                $data['unit_id'] = $this->input->post('unit_id');
                $data['strength'] = $this->input->post('strength');
                $data['medicine_name'] = $this->input->post('medicine_name');
                $data['rack_no'] = $this->input->post('rack_no');
                $data['mrp'] = $this->input->post('mrp');
                $data['purchase_price'] = $this->input->post('purchase_price');
                $data['sell_price'] = $this->input->post('sell_price');
                $data['whole_sale_price'] = $this->input->post('whole_sale_price');
                $data['minimum_stock'] = $this->input->post('minimum_stock');
                $data['note'] = $this->input->post('note');
                $data['status'] = $this->input->post('status') ? $this->input->post('status') : 0;
                $data['modified_at'] = date('Y-m-d H:i:s');
                $data['modified_by'] = logged_in_user_id();

                if ($_FILES['photo']['name']) {
                    $data['photo'] = $this->_medicine_photo();
                }

                $updated = $this->medicine->update('medicines', $data, array('id' => $this->input->post('id')));
                if ($updated) {
                    create_log('Has been updated a medicine : ' . $data['medicine_name']);
                    success($this->lang->line('update_success'));
                    redirect('pharmacy/medicine');
                } else {
                    error($this->lang->line('update_failed'));
                    redirect('pharmacy/medicine/edit/' . $this->input->post('id'));
                }
            } else {
                $this->data['medicine'] = $this->medicine->get_single('medicines', array('id' => $this->input->post('id')));
            }
        } else {
            if ($id) {
                $this->data['medicine'] = $this->medicine->get_single('medicines', array('id' => $id));
                if (!$this->data['medicine']) {
                    redirect('pharmacy/medicine');
                }
            }
        }

        $this->data['types'] = $this->medicine->get_list('medicine_types', array('status' => 1), '', '', '', 'id', 'ASC');
        $this->data['categories'] = $this->medicine->get_list('medicine_categories', array('status' => 1), '', '', '', 'id', 'ASC');
        $this->data['manufacturers'] = $this->medicine->get_list('manufacturers', array('status' => 1), '', '', '', 'id', 'ASC');
        $this->data['generics'] = $this->medicine->get_list('medicine_generics', array('status' => 1), '', '', '', 'id', 'DESC');
        $this->data['units'] = $this->medicine->get_list('medicine_units', array('status' => 1), '', '', '', 'id', 'ASC');


        $this->data['edit'] = TRUE;
        $this->layout->title($this->lang->line('edit') . ' ' . $this->lang->line('medicine') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('pharmacy/medicine/edit', $this->data);
    }

    /*     * ***************Function get_single_medicine**********************************
     * @type            : Function
     * @function name   : get_single_medicine
     * @description     : "Load single medicine information" from database                  
     *                    to the user interface   
     * @param           : null
     * @return          : null 
     * ********************************************************** */

    public function get_single_medicine()
    {
        $id = $this->input->post('id');
        $this->data['medicine'] = $this->medicine->get_single_medicine($id);
        echo $this->load->view('pharmacy/medicine/view', $this->data);
    }

    /*****************Function medicine_name**********************************
     * @type            : Function
     * @function name   : medicine_name
     * @description     : Unique check for "Medicine Name" data/value
     *
     * @param           : null
     * @return          : boolean true/false
     * ********************************************************** */
    public function medicine_name()
    {
        if ($this->input->post('id') == '') {
            $medicine = $this->medicine->duplicate_check_medicine_name($this->input->post('medicine_name'));
            if ($medicine) {
                $this->form_validation->set_message('medicine_name', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else if ($this->input->post('id') != '') {
            $medicine = $this->medicine->duplicate_check_medicine_name($this->input->post('medicine_name'), $this->input->post('id'));
            if ($medicine) {
                $this->form_validation->set_message('medicine_name', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else {
            return TRUE;
        }
    }

    /*****************Function delete**********************************
     * @type            : Function
     * @function name   : delete
     * @description     : delete "medicines" data from database
     *
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function delete($id = null)
    {

        check_permission(DELETE);

        if (!is_numeric($id)) {
            error($this->lang->line('unexpected_error'));
            redirect('pharmacy/medicine');
        }

        $medicine = $this->medicine->get_single('medicines', array('id' => $id));

        if ($this->medicine->delete('medicines', array('id' => $id))) {
            // delete medicine photo
            $destination = 'assets/uploads/';
            if (file_exists($destination . '/medicine-photo/' . $medicine->photo)) {
                @unlink($destination . '/medicine-photo/' . $medicine->photo);
            }
            create_log('Has been deleted a medicine : ' . $medicine->medicine_name);
            success($this->lang->line('delete_success'));
        } else {
            error($this->lang->line('delete_failed'));
        }
        redirect($_SERVER['HTTP_REFERER']);
    }


    /*     * ***************Function _medicine_photo**********************************
     * @type            : Function
     * @function name   : _medicine_photo
     * @description     : Process to upload medicine photo into server
     *                     and return photo name
     * @param           : null
     * @return          : $return_photo string value
     * ********************************************************** */

    private function _medicine_photo()
    {

        $prev_photo = $this->input->post('prev_photo');
        $photo = $_FILES['photo']['name'];
        $photo_type = $_FILES['photo']['type'];
        $return_photo = '';
        if ($photo != "") {
            if (
                $photo_type == 'image/jpeg' || $photo_type == 'image/pjpeg' ||
                $photo_type == 'image/jpg' || $photo_type == 'image/png' ||
                $photo_type == 'image/x-png' || $photo_type == 'image/gif'
            ) {

                $destination = 'assets/uploads/medicine-photo/';

                $file_type = explode(".", $photo);
                $extension = strtolower($file_type[count($file_type) - 1]);
                $photo_path = 'photo-' . time() . '-sbiz.' . $extension;

                move_uploaded_file($_FILES['photo']['tmp_name'], $destination . $photo_path);

                // need to unlink previous photo
                if ($prev_photo != "") {
                    if (file_exists($destination . $prev_photo)) {
                        @unlink($destination . $prev_photo);
                    }
                }

                $return_photo = $photo_path;
            }
        } else {
            $return_photo = $prev_photo;
        }

        return $return_photo;
    }

    /*****************Function addGeneric**********************************
     * @type            : Function
     * @function name   : addGeneric
     * @description     : Load "Add new Generic" user interface
     *                    and process to store "generics" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function addGeneric()
    {

        check_permission(ADD);

        $this->load->library('form_validation');
        $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
        $this->form_validation->set_rules('name', $this->lang->line('name'), 'trim|required');

        if ($this->form_validation->run() === TRUE) {
            $data['name'] = $this->input->post('name');
            $data['note'] = $this->input->post('note');
            $data['sort_by'] = 0;
            $data['status'] = 1;
            $data['created_at'] = date('Y-m-d H:i:s');
            $data['created_by'] = logged_in_user_id();

            $insert_id = $this->medicine->insert('medicine_generics', $data);

            create_log('Has been created a medicine generic : ' . $data['name']);
            $array = array('status' => 'success', 'error' => '', 'message' => $this->lang->line('insert_success'), 'id' => $insert_id, 'name' => $data['name']);
        } else {
            $msg = array(
                'name' => form_error('name')
            );
            $array = array('status' => 'failed', 'error' => $msg, 'message' => '');
        }
        echo json_encode($array);
    }

    /*****************Function price**********************************
     * @type            : Function
     * @function name   : price
     * @description     : Load "Medicine Price Update" user interface
     *                    and process to store "medicines" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function price()
    {

        check_permission(ADD);

        if ($_POST) {
            if (!empty($_POST['medicine_id'])) {
                foreach ($_POST['medicine_id'] as $key => $value) {
                    $id = $value;
                    $data['strength'] = $_POST['strength'][$value];
                    $data['rack_no'] = $_POST['rack_no'][$value];
                    $data['mrp'] = $_POST['mrp'][$value];
                    $data['purchase_price'] = $_POST['purchase_price'][$value];
                    $data['sell_price'] = $_POST['sell_price'][$value];
                    $data['whole_sale_price'] = $_POST['whole_sale_price'][$value];
                    $data['minimum_stock'] = $_POST['minimum_stock'][$value];

                    $data['modified_at'] = date('Y-m-d H:i:s');
                    $data['modified_by'] = logged_in_user_id();
                    $updated_id =  $this->medicine->update('medicines', $data, array('id' => $id));
                }

                if ($updated_id) {
                    $user =  $this->medicine->get_single('employees', array('status' => 1, 'user_id' => $this->session->userdata('user_id')));
                    create_log('Has been updated a medicine for multiple data by : ' . $user->name);
                    success($this->lang->line('update_success'));
                    redirect('pharmacy/medicine/price');
                } else {
                    error($this->lang->line('update_failed'));
                    redirect('pharmacy/medicine/price');
                }
            }
        }

        $category_id = $this->input->get('category_id');
        $manufacturer_id = $this->input->get('manufacturer_id');
        $generic_id = $this->input->get('generic_id');

        if ($_GET || $category_id || $manufacturer_id || $generic_id) {
            $this->data['medicines'] =  $this->medicine->get_medicine_price_update_list($category_id, $manufacturer_id, $generic_id);

            $this->data['category_id'] = $category_id;
            $this->data['manufacturer_id'] = $manufacturer_id;
            $this->data['generic_id'] = $generic_id;
        }

        $this->data['categories'] = $this->medicine->get_list('medicine_categories', array('status' => 1), '', '', '', 'id', 'ASC');
        $this->data['manufacturers'] = $this->medicine->get_list('manufacturers', array('status' => 1), '', '', '', 'id', 'ASC');
        $this->data['generics'] = $this->medicine->get_list('medicine_generics', array('status' => 1), '', '', '', 'id', 'ASC');

        $this->data['add'] = TRUE;
        $this->layout->title($this->lang->line('price') . ' ' . $this->lang->line('update') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('pharmacy/medicine/price', $this->data);
    }
}
