<?php

defined('BASEPATH') or exit('No direct script access allowed');

/* * *****************Manufacturer.php**********************************
 * @product name    : Hospital Management System
 * @type            : Class
 * @class name      : Manufacturer
 * @description     : Manage manufacturer company.
 * @author          : South Bangla IT Zone Team
 * @url             : http://hms.sbitzone.com
 * @support         : rabibd.it@gmail.com
 * @copyright       : South Bangla IT Zone Team
 * ********************************************************** */

class Manufacturer extends MY_Controller {

    public $data = array();

    function __construct() {
        parent::__construct();
        $this->load->model('Manufacturer_Model', 'manufacturer', true);
    }

    /*****************Function index**********************************
     * @type            : Function
     * @function name   : index
     * @description     : Load "Manufacturer List" user interface
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function index() {

        check_permission(VIEW);

        $keyword = trim($this->input->get('keyword'));
        $config = array();
        $config = get_pagination(ADMIN_PER_PAGE);
        if ($keyword) {
            $config['base_url'] = site_url('pharmacy/manufacturer/?search=') . $keyword;
        } else {
            $config['base_url'] = current_url();
        }
        if ($this->input->get('page')) {
            $sgm = (int) trim($this->input->get('page'));
            $offset = $config['per_page'] * ($sgm - 1);
        } else {
            $offset = 0;
        }

        $config['total_rows'] = $this->manufacturer->get_manufacturer_list($config['per_page'], $offset, $keyword, $count = TRUE);
        $this->pagination->initialize($config);
        $this->data['manufacturers'] = $this->manufacturer->get_manufacturer_list($config['per_page'], $offset, $keyword, $count = FALSE);
        $this->data['sl_count'] = ($this->input->get('page') == 0 ? 1 : (($this->input->get('page') - 1) * $config["per_page"] + 1));

        $this->data['keyword'] = $keyword;
        $this->data['list'] = TRUE;
        $this->layout->title($this->lang->line('manage_manufacturer') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('manufacturer/index', $this->data);
    }

    /*****************Function add**********************************
     * @type            : Function
     * @function name   : add
     * @description     : Load "Add new Manufacturer" user interface
     *                    and process to store "manufacturers" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function add() {

        check_permission(ADD);

        if ($_POST) {
            $this->_prepare_manufacturer_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_manufacturer_data();
                $insert_id = $this->manufacturer->insert('manufacturers', $data);
                if ($insert_id) {
                    create_log('Has been created a Manufacturer : ' . $data['name']);
                    success($this->lang->line('insert_success'));
                    redirect('pharmacy/manufacturer');
                } else {
                    error($this->lang->line('insert_failed'));
                    redirect('pharmacy/manufacturer/add');
                }
            } else {
                $this->data = $_POST;
            }
        }
        $this->data['add'] = TRUE;
        $this->layout->title($this->lang->line('add') . ' ' . $this->lang->line('manufacturer') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('manufacturer/add', $this->data);
    }

    /*****************Function edit**********************************
     * @type            : Function
     * @function name   : edit
     * @description     : Load Update "Manufacturer" user interface
     *                    with populate "Manufacturer" value
     *                    and process to update "manufacturers" into database
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function edit($id = null) {

        check_permission(EDIT);

        if ($_POST) {
            $this->_prepare_manufacturer_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_manufacturer_data();
                $updated = $this->manufacturer->update('manufacturers', $data, array('id' => $this->input->post('id')));
                if ($updated) {
                    create_log('Has been updated a Manufacturer : ' . $data['name']);
                    success($this->lang->line('update_success'));
                    redirect('pharmacy/manufacturer');
                } else {
                    error($this->lang->line('update_failed'));
                    redirect('pharmacy/manufacturer/edit/' . $this->input->post('id'));
                }
            } else {
                $this->data['manufacturer'] = $this->manufacturer->get_single('manufacturers', array('id' => $this->input->post('id')));
            }
        } else {
            if ($id) {
                $this->data['manufacturer'] = $this->manufacturer->get_single('manufacturers', array('id' => $id));
                if (!$this->data['manufacturer']) {
                    redirect('pharmacy/manufacturer');
                }
            }
        }
        $this->data['edit'] = TRUE;
        $this->layout->title($this->lang->line('edit') . ' ' . $this->lang->line('manufacturer') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('manufacturer/edit', $this->data);
    }

    /*     * ***************Function get_single_manufacturer**********************************
     * @type            : Function
     * @function name   : get_single_manufacturer
     * @description     : "Load single manufacturer information" from database                  
     *                    to the user interface   
     * @param           : null
     * @return          : null 
     * ********************************************************** */

    public function get_single_manufacturer() {
        $id = $this->input->post('id');
        $this->data['manufacturer'] = $this->manufacturer->get_single_manufacturer($id);
        echo $this->load->view('manufacturer/view', $this->data);
    }

    /*****************Function _prepare_manufacturer_validation**********************************
     * @type            : Function
     * @function name   : _prepare_manufacturer_validation
     * @description     : Process "Manufacturer" user input data validation
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    private function _prepare_manufacturer_validation() {
        $this->load->library('form_validation');
        $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
        $this->form_validation->set_rules('name', $this->lang->line('manufacturer') . ' ' . $this->lang->line('name'), 'trim|required|callback_name');
    }

    /*****************Function name**********************************
     * @type            : Function
     * @function name   : name
     * @description     : Unique check for "Manufacturer Name" data/value
     *
     * @param           : null
     * @return          : boolean true/false
     * ********************************************************** */
    public function name() {
        if ($this->input->post('id') == '') {
            $manufacturer = $this->manufacturer->duplicate_check($this->input->post('name'));
            if ($manufacturer) {
                $this->form_validation->set_message('name', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else if ($this->input->post('id') != '') {
            $manufacturer = $this->manufacturer->duplicate_check($this->input->post('name'), $this->input->post('id'));
            if ($manufacturer) {
                $this->form_validation->set_message('name', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else {
            return TRUE;
        }
    }

    /*****************Function _get_posted_manufacturer_data**********************************
     * @type            : Function
     * @function name   : _get_posted_manufacturer_data
     * @description     : Prepare "Manufacturer" user input data to save into database
     *
     * @param           : null
     * @return          : $data array(); value
     * ********************************************************** */
    private function _get_posted_manufacturer_data() {

        $items = array();
        $items[] = 'name';
        $items[] = 'phone';
        $items[] = 'email';
        $items[] = 'fax';
        $items[] = 'web';
        $items[] = 'address';
        $items[] = 'established';
        $items[] = 'market_share';
        $items[] = 'growth';
        $items[] = 'headquarter';
        $items[] = 'google_map';
        $items[] = 'about_us_en';
        $items[] = 'about_us_bn';
        $items[] = 'note';
        $items[] = 'sort_by';
        $items[] = 'status';
        $data = elements($items, $_POST);

        if ($this->input->post('id')) {
            $data['modified_at'] = date('Y-m-d H:i:s');
            $data['modified_by'] = logged_in_user_id();
        } else {
            $data['created_at'] = date('Y-m-d H:i:s');
            $data['created_by'] = logged_in_user_id();
        }
        return $data;
    }

    /*****************Function delete**********************************
     * @type            : Function
     * @function name   : delete
     * @description     : delete "Manufacturer" data from database
     *
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function delete($id = null) {

        check_permission(DELETE);

        if (!is_numeric($id)) {
            error($this->lang->line('unexpected_error'));
            redirect('pharmacy/manufacturer');
        }

        $manufacturer = $this->manufacturer->get_single('manufacturers', array('id' => $id));

        if ($this->manufacturer->delete('manufacturers', array('id' => $id))) {
            create_log('Has been deleted a Manufacturer : ' . $manufacturer->name);
            success($this->lang->line('delete_success'));
        } else {
            error($this->lang->line('delete_failed'));
        }
        redirect('pharmacy/manufacturer');
    }
}
