<?php

defined('BASEPATH') or exit('No direct script access allowed');

/* * *****************Medicine.php**********************************
 * @product name    : Hospital Management System
 * @type            : Class
 * @class name      : Medicine
 * @description     : Manage IPD Medicine Chart.
 * @author          : South Bangla IT Zone Team
 * @url             : http://hms.sbitzone.com
 * @support         : rabibd.it@gmail.com
 * @copyright       : South Bangla IT Zone Team
 * ********************************************************** */

class Medicine extends MY_Controller {

    public $data = array();

    function __construct() {
        parent::__construct();
        $this->load->model('Medicine_Model', 'medicine', true);
        $this->load->model('Ipd_Model', 'ipd', true);
        $this->data['doctors'] = $this->ipd->get_doctor_list();
        $this->data['nurses'] = $this->ipd->get_nurse_list();
    }

    public function index($ipd_id) {

        check_permission(VIEW);

        $this->data['patient'] = $this->ipd->get_single_ipd_patient($ipd_id);
        $this->data['total_investigation_due'] = $this->ipd->get_total_investigation_due($this->data['patient']->patient_id);
        $this->data['total_pharmacy_due'] = $this->ipd->get_total_pharmacy_due($this->data['patient']->patient_id);
        $this->data['total_ipd_advanced'] = $this->ipd->get_total_ipd_advaced($ipd_id, $this->data['patient']->patient_id);
        

        $search = trim($this->input->get('search'));
        $config = array();
        $config = get_pagination(ADMIN_PER_PAGE);
        if ($search) {
            $config['base_url'] = site_url('ipd/medicine/index/' . $ipd_id . '/?search=' . $search);
        } else {
            $config['base_url'] = current_url();
        }
        if ($this->input->get('page')) {
            $sgm = (int) trim($this->input->get('page'));
            $offset = $config['per_page'] * ($sgm - 1);
        } else {
            $offset = 0;
        }

        $config['total_rows'] = $this->medicine->get_medicine_chart_list($config['per_page'], $offset, $ipd_id, $search, $count = TRUE);
        $this->pagination->initialize($config);
        $this->data['medicines'] = $this->medicine->get_medicine_chart_list($config['per_page'], $offset, $ipd_id, $search, $count = FALSE);
        $this->data['sl_count'] = ($this->input->get('page') == 0 ? 1 : (($this->input->get('page') - 1) * $config["per_page"] + 1));
        $this->data['search'] = $search;


        $this->data['list'] = TRUE;
        $this->layout->title($this->lang->line('medicine') . ' ' . $this->lang->line('chart') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('medicine/index', $this->data);
    }

    /*****************Function add**********************************
     * @type            : Function
     * @function name   : add
     * @description     : Load "Add Medicine" user interface
     *                    and process to store "ipd_medicine_charts" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */

    public function add() {

        check_permission(ADD);

        if ($_POST) {
            $this->load->library('form_validation');
            $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
            $this->form_validation->set_rules('doctor_id', $this->lang->line('doctor') . ' ' . $this->lang->line('name'), 'trim|required');
            $this->form_validation->set_rules('date', $this->lang->line('date') . ' ' . $this->lang->line('name'), 'trim|required');
            if ($this->form_validation->run() === TRUE) {
                $data['ipd_id'] = $this->input->post('ipd_id');
                $data['patient_id'] = $this->input->post('patient_id');
                $data['doctor_id'] = $this->input->post('doctor_id');
                $data['date'] = date('Y-m-d H:i', strtotime($this->input->post('date')));
                $data['dose'] = $this->input->post('dose');
                $data['route'] = $this->input->post('route');
                $data['medicine_name'] = $this->input->post('medicine_name');
                $data['nurse_id'] = $this->input->post('nurse_id');
                $data['medicine_status'] = $this->input->post('medicine_status');
                $data['note'] = $this->input->post('note');
                $data['status'] = '1';
                $data['created_at'] = date('Y-m-d H:i:s');
                $data['created_by'] = logged_in_user_id();
                $insert_id = $this->medicine->insert('ipd_medicine_charts', $data);
                if ($insert_id) {
                    $patient = $this->medicine->get_single('patients', array('user_id' => $data['patient_id']));
                    create_log('Has been created a medicine chart : ' . $patient->name);
                    success($this->lang->line('insert_success'));
                    redirect('ipd/medicine/index/' . $data['ipd_id']);
                } else {
                    error($this->lang->line('insert_failed'));
                    redirect('ipd/medicine/index/' . $data['ipd_id']);
                }
            } else {
                $this->data = $_POST;
            }
        }
    }

    /*****************Function get_medicine_chart**********************************
     * @type            : Function
     * @function name   : get_medicine_chart
     * @description     : Load "Single Medicine Chart" user interface
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function get_medicine_chart($id) {
        if ($id > 0) {
            $medicine = $this->medicine->get_single('ipd_medicine_charts', array('id' => $id));
            echo (json_encode($medicine));
        }
    }

    /*****************Function edit**********************************
     * @type            : Function
     * @function name   : edit
     * @description     : Load Update "Medicine Chart" user interface
     *                    with populate "Medicine Chart" value
     *                    and process to update "ipd_medicine_charts" into database
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function edit($id = null) {

        check_permission(EDIT);

        if ($_POST) {

            $this->load->library('form_validation');
            $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
            $this->form_validation->set_rules('doctor_id', $this->lang->line('doctor') . ' ' . $this->lang->line('name'), 'trim|required');
            $this->form_validation->set_rules('date', $this->lang->line('date') . ' ' . $this->lang->line('name'), 'trim|required');
            if ($this->form_validation->run() === TRUE) {
                $data['doctor_id'] = $this->input->post('doctor_id');
                $data['date'] = date('Y-m-d H:i', strtotime($this->input->post('date')));
                $data['dose'] = $this->input->post('dose');
                $data['route'] = $this->input->post('route');
                $data['medicine_name'] = $this->input->post('medicine_name');
                $data['nurse_id'] = $this->input->post('nurse_id');
                $data['medicine_status'] = $this->input->post('medicine_status');
                $data['note'] = $this->input->post('note');
                $data['modified_at'] = date('Y-m-d H:i:s');
                $data['modified_by'] = logged_in_user_id();

                $this->medicine->update('ipd_medicine_charts', $data, array('id' => $this->input->post('medicine_id')));
                $patient = $this->medicine->get_single('patients', array('user_id' => $this->input->post('patient_id')));
                create_log('Has been updated a medicine chart : ' . $patient->name);
                $array = array('status' => 'success', 'error' => '', 'message' => $this->lang->line('update_success'));
            } else {
                $msg = array(
                    'doctor_id' => form_error('doctor_id'),
                    'date' => form_error('date'),
                );
                $array = array('status' => 'failed', 'error' => $msg, 'message' => '');
            }

            echo json_encode($array);
        }
    }

    /*     * ***************Function get_single_medicine_chart**********************************
     * @type            : Function
     * @function name   : get_single_medicine_chart
     * @description     : "Load single medicine chart information" from database                  
     *                    to the user interface   
     * @param           : null
     * @return          : null 
     * ********************************************************** */

    public function get_single_medicine_chart() {
        $id = $this->input->post('id');
        $this->data['medicine'] = $this->medicine->get_single_medicine_chart($id);
        echo $this->load->view('ipd/medicine/view', $this->data);
    }

    /*****************Function delete**********************************
     * @type            : Function
     * @function name   : delete
     * @description     : delete "Medicine Chart" data from database
     *
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function delete($id = null) {

        check_permission(DELETE);

        if (!is_numeric($id)) {
            error($this->lang->line('unexpected_error'));
            redirect($_SERVER['HTTP_REFERER']);
        }

        $medicine = $this->medicine->get_single('ipd_medicine_charts', array('id' => $id));
        $patient = $this->medicine->get_single('patients', array('user_id' => $medicine->patient_id));

        if ($this->ipd->delete('ipd_medicine_charts', array('id' => $id))) {
            create_log('Has been deleted a medicine chart : ' . $patient->name);
            success($this->lang->line('delete_success'));
        } else {
            error($this->lang->line('delete_failed'));
        }
        redirect($_SERVER['HTTP_REFERER']);
    }
}
