<?php

defined('BASEPATH') or exit('No direct script access allowed');

/* * *****************Bed.php**********************************
 * @product name    : Hospital Management System
 * @type            : Class
 * @class name      : Bed
 * @description     : Manage IPD Bed Chart.
 * @author          : South Bangla IT Zone Team
 * @url             : http://hms.sbitzone.com
 * @support         : rabibd.it@gmail.com
 * @copyright       : South Bangla IT Zone Team
 * ********************************************************** */

class Bed extends MY_Controller {

    public $data = array();

    function __construct() {
        parent::__construct();
        $this->load->model('Bed_Model', 'bed', true);
        $this->load->model('Ipd_Model', 'ipd', true);
        $this->data['bed_groups'] = $this->ipd->get_bed_group_list();
        $this->data['bed_list'] = $this->ipd->get_list('beds', array('status' => 1, 'is_active' => 0), '', '', '', 'id', 'ASC');
    }

    public function index($ipd_id) {

        check_permission(VIEW);

        $this->data['patient'] = $this->ipd->get_single_ipd_patient($ipd_id);
        $this->data['total_investigation_due'] = $this->ipd->get_total_investigation_due($this->data['patient']->patient_id);
        $this->data['total_pharmacy_due'] = $this->ipd->get_total_pharmacy_due($this->data['patient']->patient_id);
        $this->data['total_ipd_advanced'] = $this->ipd->get_total_ipd_advaced($ipd_id, $this->data['patient']->patient_id);
        

        $search = trim($this->input->get('search'));
        $config = array();
        $config = get_pagination(ADMIN_PER_PAGE);
        if ($search) {
            $config['base_url'] = site_url('ipd/bed/index/' . $ipd_id . '/?search=' . $search);
        } else {
            $config['base_url'] = current_url();
        }
        if ($this->input->get('page')) {
            $sgm = (int) trim($this->input->get('page'));
            $offset = $config['per_page'] * ($sgm - 1);
        } else {
            $offset = 0;
        }

        $config['total_rows'] = $this->bed->get_patient_bed_list($config['per_page'], $offset, $ipd_id, $search, $count = TRUE);
        $this->pagination->initialize($config);
        $this->data['beds'] = $this->bed->get_patient_bed_list($config['per_page'], $offset, $ipd_id, $search, $count = FALSE);
        $this->data['sl_count'] = ($this->input->get('page') == 0 ? 1 : (($this->input->get('page') - 1) * $config["per_page"] + 1));
        $this->data['search'] = $search;

        $this->data['list'] = TRUE;
        $this->layout->title($this->lang->line('bed') . ' ' . $this->lang->line('chart') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('bed/index', $this->data);
    }

    /*****************Function add**********************************
     * @type            : Function
     * @function name   : add
     * @description     : Load "Add Bed" user interface
     *                    and process to store "ipd_beds" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */

    public function add() {

        check_permission(ADD);

        if ($_POST) {
            $this->load->library('form_validation');
            $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
            $this->form_validation->set_rules('bed_group_id', $this->lang->line('bed') . ' ' . $this->lang->line('group'), 'trim|required');
            $this->form_validation->set_rules('bed_id', $this->lang->line('bed') . ' ' . $this->lang->line('name'), 'trim|required');
            $this->form_validation->set_rules('from_date', $this->lang->line('from_date'), 'trim|required');
            if ($this->form_validation->run() === TRUE) {
                $data['ipd_id'] = $this->input->post('ipd_id');
                $data['patient_id'] = $this->input->post('patient_id');
                $data['bed_group_id'] = $this->input->post('bed_group_id');
                $data['bed_id'] = $this->input->post('bed_id');
                $data['from_date'] = date('Y-m-d H:i', strtotime($this->input->post('from_date')));
                $data['created_at'] = date('Y-m-d H:i:s');
                $data['created_by'] = logged_in_user_id();

                $to_date =  date('Y-m-d H:i:s', strtotime("-1 day", strtotime($this->input->post('from_date'))));;
                $bed = $this->bed->get_single('ipd_beds', array('ipd_id' => $data['ipd_id'], 'patient_id' => $data['patient_id']), '', '', '', 'id', 'DESC');
                $this->bed->update('ipd_beds', array('to_date' => $to_date), array('id' => $bed->id));
                $this->bed->update('beds', array('is_active' => 0), array('id' => $bed->bed_id));

                $insert_id = $this->bed->insert('ipd_beds', $data);
                if ($insert_id) {
                    $this->bed->update('beds', array('is_active' => 1), array('id' => $data['bed_id']));
                    $this->bed->update('ipd_patients', array('bed_group_id' => $data['bed_group_id'], 'bed_id' => $data['bed_id']), array('id' => $data['ipd_id']));

                    $patient = $this->bed->get_single('patients', array('user_id' => $data['patient_id']));
                    create_log('Has been created a bed : ' . $patient->name);
                    success($this->lang->line('insert_success'));
                    redirect('ipd/bed/index/' . $data['ipd_id']);
                } else {
                    error($this->lang->line('insert_failed'));
                    redirect('ipd/bed/index/' . $data['ipd_id']);
                }
            } else {
                $msg = array(
                    'bed_group_id' => form_error('bed_group_id'),
                    'bed_id' => form_error('bed_id'),
                    'from_date' => form_error('from_date')
                );
                $array = array('status' => 'failed', 'error' => $msg, 'message' => '');
            }
            echo json_encode($array);
        }
    }

    /*****************Function get_bed**********************************
     * @type            : Function
     * @function name   : get_bed
     * @description     : Load "Single Bed" user interface
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function get_bed($id) {
        if ($id > 0) {
            $bed = $this->bed->get_single('ipd_beds', array('id' => $id));
            echo (json_encode($bed));
        }
    }

    /*****************Function edit**********************************
     * @type            : Function
     * @function name   : edit
     * @description     : Load Update "Bed Chart" user interface
     *                    with populate "Bed Chart" value
     *                    and process to update "ipd_bed_visits" into database
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function edit() {

        check_permission(EDIT);

        if ($_POST) {
            $this->load->library('form_validation');
            $this->form_validation->set_rules('bed_group_id', $this->lang->line('bed') . ' ' . $this->lang->line('group'), 'trim|required');
            $this->form_validation->set_rules('bed_id', $this->lang->line('bed') . ' ' . $this->lang->line('name'), 'trim|required');
            $this->form_validation->set_rules('from_date', $this->lang->line('from_date'), 'trim|required');
            if ($this->form_validation->run() === TRUE) {
                $data['ipd_id'] = $this->input->post('ipd_id');
                $data['patient_id'] = $this->input->post('patient_id');
                $data['bed_group_id'] = $this->input->post('bed_group_id');
                $data['bed_id'] = $this->input->post('bed_id');
                $data['from_date'] = date('Y-m-d H:i', strtotime($this->input->post('from_date')));
                $data['note'] = $this->input->post('note');
                $data['modified_at'] = date('Y-m-d H:i:s');
                $data['modified_by'] = logged_in_user_id();

                $to_date =  date('Y-m-d H:i:s', strtotime("-1 day", strtotime($this->input->post('from_date'))));;
                $bed = $this->bed->get_single('ipd_beds', array('ipd_id' => $data['ipd_id'], 'patient_id' => $data['patient_id']), '', '', '', 'id', 'DESC');
                $this->bed->update('ipd_beds', array('to_date' => $to_date), array('id' => $bed->id));
                $this->bed->update('beds', array('is_active' => 0), array('id' => $bed->bed_id));

                $updated_id = $this->bed->update('ipd_beds', $data, array('id' => $this->input->post('id')));
                if ($updated_id) {
                    $this->bed->update('beds', array('is_active' => 1), array('id' => $data['bed_id']));
                    $this->bed->update('ipd_patients', array('bed_group_id' => $data['bed_group_id'], 'bed_id' => $data['bed_id']), array('id' => $data['ipd_id']));

                    $patient = $this->bed->get_single('patients', array('user_id' => $data['patient_id']));
                    create_log('Has been updated a bed : ' . $patient->name);
                    $array = array('status' => 'success', 'error' => '', 'message' => $this->lang->line('update_success'));
                }
            } else {
                $msg = array(
                    'bed_group_id' => form_error('bed_group_id'),
                    'bed_id' => form_error('bed_id'),
                    'from_date' => form_error('from_date')
                );
                $array = array('status' => 'failed', 'error' => $msg, 'message' => '');
            }
            echo json_encode($array);
        }
    }

    /*     * ***************Function get_single_bed**********************************
     * @type            : Function
     * @function name   : get_single_bed
     * @description     : "Load single bed information" from database                  
     *                    to the user interface   
     * @param           : null
     * @return          : null 
     * ********************************************************** */

    public function get_single_bed() {
        $id = $this->input->post('id');
        $this->data['bed'] = $this->bed->get_single_bed($id);
        echo $this->load->view('ipd/bed/view', $this->data);
    }

    /*****************Function delete**********************************
     * @type            : Function
     * @function name   : delete
     * @description     : delete "Bed" data from database
     *
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function delete($id = null) {

        check_permission(DELETE);

        if (!is_numeric($id)) {
            error($this->lang->line('unexpected_error'));
            redirect($_SERVER['HTTP_REFERER']);
        }

        $bed = $this->bed->get_single('ipd_beds', array('id' => $id));
        $patient = $this->bed->get_single('patients', array('user_id' => $bed->patient_id));

        if ($this->ipd->delete('ipd_beds', array('id' => $id))) {
            create_log('Has been deleted a bed : ' . $patient->name);
            success($this->lang->line('delete_success'));
        } else {
            error($this->lang->line('delete_failed'));
        }
        redirect($_SERVER['HTTP_REFERER']);
    }
}
