<?php

defined('BASEPATH') or exit('No direct script access allowed');

/* * *****************Anesthesia.php**********************************
 * @product name    : Hospital Management System
 * @type            : Class
 * @class name      : Anesthesia
 * @description     : Manage IPD Anesthesia Chart.
 * @author          : South Bangla IT Zone Team
 * @url             : http://hms.sbitzone.com
 * @support         : rabibd.it@gmail.com
 * @copyright       : South Bangla IT Zone Team
 * ********************************************************** */

class Anesthesia extends MY_Controller {

    public $data = array();

    function __construct() {
        parent::__construct();
        $this->load->model('Anesthesia_Model', 'anesthesia', true);
        $this->load->model('Ipd_Model', 'ipd', true);
        $this->data['doctors'] = $this->anesthesia->get_doctor_list();
    }

    public function index($ipd_id) {

        check_permission(VIEW);

        $this->data['patient'] = $this->ipd->get_single_ipd_patient($ipd_id);
        $this->data['total_investigation_due'] = $this->ipd->get_total_investigation_due($this->data['patient']->patient_id);
        $this->data['total_pharmacy_due'] = $this->ipd->get_total_pharmacy_due($this->data['patient']->patient_id);
        $this->data['total_ipd_advanced'] = $this->ipd->get_total_ipd_advaced($ipd_id, $this->data['patient']->patient_id);
        

        $search = trim($this->input->get('search'));
        $config = array();
        $config = get_pagination(ADMIN_PER_PAGE);
        if ($search) {
            $config['base_url'] = site_url('ipd/anesthesia/index/' . $ipd_id . '/?search=' . $search);
        } else {
            $config['base_url'] = current_url();
        }
        if ($this->input->get('page')) {
            $sgm = (int) trim($this->input->get('page'));
            $offset = $config['per_page'] * ($sgm - 1);
        } else {
            $offset = 0;
        }

        $config['total_rows'] = $this->anesthesia->get_patient_anesthesia_list($config['per_page'], $offset, $ipd_id, $search, $count = TRUE);
        $this->pagination->initialize($config);
        $this->data['anesthesias'] = $this->anesthesia->get_patient_anesthesia_list($config['per_page'], $offset, $ipd_id, $search, $count = FALSE);
        $this->data['sl_count'] = ($this->input->get('page') == 0 ? 1 : (($this->input->get('page') - 1) * $config["per_page"] + 1));
        $this->data['search'] = $search;

        $this->data['list'] = TRUE;
        $this->layout->title($this->lang->line('anesthesia') . ' ' . $this->lang->line('chart') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('anesthesia/index', $this->data);
    }

    /*****************Function add**********************************
     * @type            : Function
     * @function name   : add
     * @description     : Load "Add Anesthesia" user interface
     *                    and process to store "ipd_anesthesias" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */

    public function add() {

        check_permission(ADD);

        if ($_POST) {
            $this->load->library('form_validation');
            $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
            $this->form_validation->set_rules('doctor_id', $this->lang->line('anesthesia') . ' ' . $this->lang->line('name'), 'trim|required');
            $this->form_validation->set_rules('charge', $this->lang->line('doctor') . ' ' . $this->lang->line('charge'), 'trim|required');
            if ($this->form_validation->run() === TRUE) {
                $data['ipd_id'] = $this->input->post('ipd_id');
                $data['patient_id'] = $this->input->post('patient_id');
                $data['type'] = 'anesthesia';
                $data['doctor_id'] = $this->input->post('doctor_id');
                $data['charge'] = $this->input->post('charge');
                $data['date'] = date('Y-m-d H:i', strtotime($this->input->post('date')));
                $data['instruction'] = $this->input->post('instruction');
                $data['anesthesia_status'] = 'complete';
                $data['created_at'] = date('Y-m-d H:i:s');
                $data['created_by'] = logged_in_user_id();
                $insert_id = $this->anesthesia->insert('ipd_anesthesias', $data);
                if ($insert_id) {
                    $patient = $this->anesthesia->get_single('patients', array('user_id' => $data['patient_id']));
                    create_log('Has been created a anesthesia : ' . $patient->name);
                    success($this->lang->line('insert_success'));
                    redirect('ipd/anesthesia/index/' . $data['ipd_id']);
                } else {
                    error($this->lang->line('insert_failed'));
                    redirect('ipd/anesthesia/index/' . $data['ipd_id']);
                }
            } else {
                $this->data = $_POST;
            }
        }
    }

    /*****************Function get_anesthesia**********************************
     * @type            : Function
     * @function name   : get_anesthesia
     * @description     : Load "Single Anesthesia" user interface
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function get_anesthesia($id) {
        if ($id > 0) {
            $anesthesia = $this->anesthesia->get_single('ipd_anesthesias', array('id' => $id));
            echo (json_encode($anesthesia));
        }
    }

    /*****************Function edit**********************************
     * @type            : Function
     * @function name   : edit
     * @description     : Load Update "Anesthesia Chart" user interface
     *                    with populate "Anesthesia Chart" value
     *                    and process to update "ipd_anesthesias" into database
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function edit() {

        check_permission(EDIT);

        if ($_POST) {
            $this->load->library('form_validation');
            $this->form_validation->set_rules('doctor_id', $this->lang->line('anesthesia') . ' ' . $this->lang->line('name'), 'trim|required');
            $this->form_validation->set_rules('charge', $this->lang->line('doctor') . ' ' . $this->lang->line('charge'), 'trim|required');
            if ($this->form_validation->run() === TRUE) {
                $data['type'] = 'anesthesia';
                $data['doctor_id'] = $this->input->post('doctor_id');
                $data['charge'] = $this->input->post('charge');
                $data['date'] = date('Y-m-d H:i', strtotime($this->input->post('date')));
                $data['instruction'] = $this->input->post('instruction');
                $data['anesthesia_status'] = $this->input->post('anesthesia_status');
                $data['note'] = $this->input->post('note');
                $data['modified_at'] = date('Y-m-d H:i:s');
                $data['modified_by'] = logged_in_user_id();

                $this->anesthesia->update('ipd_anesthesias', $data, array('id' => $this->input->post('anesthesia_id')));
                $patient = $this->anesthesia->get_single('patients', array('user_id' => $this->input->post('patient_id')));
                create_log('Has been updated a anesthesia : ' . $patient->name);
                $array = array('status' => 'success', 'error' => '', 'message' => $this->lang->line('update_success'));
            } else {
                $msg = array(
                    'doctor_id' => form_error('doctor_id'),
                    'date' => form_error('date'),
                );
                $array = array('status' => 'failed', 'error' => $msg, 'message' => '');
            }

            echo json_encode($array);
        }
    }

    /*     * ***************Function get_single_anesthesia**********************************
     * @type            : Function
     * @function name   : get_single_anesthesia
     * @description     : "Load single anesthesia information" from database                  
     *                    to the user interface   
     * @param           : null
     * @return          : null 
     * ********************************************************** */

    public function get_single_anesthesia() {
        $id = $this->input->post('id');
        $this->data['anesthesia'] = $this->anesthesia->get_single_anesthesia($id);
        echo $this->load->view('ipd/anesthesia/view', $this->data);
    }

    /*****************Function get_doctor_charge**********************************
     * @type            : Function
     * @function name   : get_doctor_charge
     * @description     : Load "IPD Doctor Charge" user interface
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function get_doctor_charge() {
        $user_id = $this->input->post('user_id');
        if ($user_id > 0) {
            $doctor_charges = $this->anesthesia->get_single('anesthetists', array('user_id' => $user_id));
            $doctor_charge = array(
                'charge' => $doctor_charges->charge
            );
        }
        echo (json_encode($doctor_charge));
    }

    /*****************Function delete**********************************
     * @type            : Function
     * @function name   : delete
     * @description     : delete "Anesthesia" data from database
     *
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function delete($id = null) {

        check_permission(DELETE);

        if (!is_numeric($id)) {
            error($this->lang->line('unexpected_error'));
            redirect($_SERVER['HTTP_REFERER']);
        }

        $anesthesia = $this->anesthesia->get_single('ipd_anesthesias', array('id' => $id));
        $patient = $this->anesthesia->get_single('patients', array('user_id' => $anesthesia->patient_id));

        if ($this->ipd->delete('ipd_anesthesias', array('id' => $id))) {
            create_log('Has been deleted a anesthesia : ' . $patient->name);
            success($this->lang->line('delete_success'));
        } else {
            error($this->lang->line('delete_failed'));
        }
        redirect($_SERVER['HTTP_REFERER']);
    }
}
