<?php

defined('BASEPATH') or exit('No direct script access allowed');

/* * *****************Diagnostic.php**********************************
 * @product name    : Hospital Management System
 * @type            : Class
 * @class name      : Diagnostic
 * @description     : Manage diagnostic.
 * @author          : South Bangla IT Zone Team
 * @url             : http://hms.sbitzone.com
 * @support         : rabibd.it@gmail.com
 * @copyright       : South Bangla IT Zone Team
 * ********************************************************** */

class Diagnostic extends MY_Controller {

    public $data = array();

    function __construct() {
        parent::__construct();
        $this->load->model('Diagnostic_Model', 'diagnostic', true);
    }


    /*****************Function index**********************************
     * @type            : Function
     * @function name   : index
     * @description     : Load "department List" user interface
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function index() {

        check_permission(VIEW);

        $this->data['total'] = $this->diagnostic->total_rate();
        $this->data['list'] = TRUE;
        $this->layout->title($this->lang->line('manage_diagnostic') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('diagnostic/index', $this->data);
    }

    /*****************Function department**********************************
     * @type            : Function
     * @function name   : department
     * @description     : Load "department List" user interface
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function department() {

        check_permission(VIEW);

        $search = $this->input->get('search');

        $config = array();
        $config = get_pagination(ADMIN_PER_PAGE);
        if ($search) {
            $config['base_url'] = site_url('hospital/diagnostic/department/?search=') . $search;
        } else {
            $config['base_url'] = current_url();
        }
        if ($this->input->get('page')) {
            $sgm = (int) trim($this->input->get('page'));
            $offset = $config['per_page'] * ($sgm - 1);
        } else {
            $offset = 0;
        }

        $config['total_rows'] = $this->diagnostic->get_department_list($config['per_page'], $offset, $search, $count = TRUE);
        $this->pagination->initialize($config);
        $this->data['departments'] = $this->diagnostic->get_department_list($config['per_page'], $offset, $search, $count = FALSE);
        $this->data['sl_count'] = ($this->input->get('page') == 0 ? 1 : (($this->input->get('page') - 1) * $config["per_page"] + 1));

        $this->data['search'] = $search;
        $this->data['list'] = TRUE;
        $this->layout->title($this->lang->line('manage_department') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('diagnostic/department/index', $this->data);
    }

    /*****************Function add_department**********************************
     * @type            : Function
     * @function name   : add_department
     * @description     : Load "Add new department" user interface
     *                    and process to store "diagnostic_departments" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function add_department() {

        check_permission(ADD);

        if ($_POST) {

            $this->load->library('form_validation');
            $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
            $this->form_validation->set_rules('name', $this->lang->line('name'), 'trim|required|callback_department_name');
            $this->form_validation->set_rules('bn_name', $this->lang->line('bn_name'), 'trim');

            if ($this->form_validation->run() === TRUE) {
                $data['name'] = $this->input->post('name');
                $data['bn_name'] = $this->input->post('bn_name');
                $data['note'] = $this->input->post('note');
                $data['sort_by'] = $this->input->post('sort_by');
                $data['status'] = $this->input->post('status') ? $this->input->post('status') : 0;
                $data['created_at'] = date('Y-m-d H:i:s');
                $data['created_by'] = logged_in_user_id();

                $insert_id = $this->diagnostic->insert('diagnostic_departments', $data);
                if ($insert_id) {
                    create_log('Has been created a diagnostic department : ' . $data['name']);
                    success($this->lang->line('insert_success'));
                    redirect('hospital/diagnostic/department');
                } else {
                    error($this->lang->line('insert_failed'));
                    redirect('hospital/diagnostic/add_department');
                }
            } else {
                $this->data = $_POST;
            }
        }

        $this->data['add'] = TRUE;
        $this->layout->title($this->lang->line('add') . ' ' . $this->lang->line('department') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('diagnostic/department/add', $this->data);
    }

    /*****************Function edit_department**********************************
     * @type            : Function
     * @function name   : edit_department
     * @description     : Load "Edit new department" user interface
     *                    and process to store "diagnostic_departments" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function edit_department($id = NULL) {

        check_permission(EDIT);

        if ($_POST) {

            $this->load->library('form_validation');
            $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
            $this->form_validation->set_rules('name', $this->lang->line('name'), 'trim|required|callback_department_name');
            $this->form_validation->set_rules('bn_name', $this->lang->line('bn_name'), 'trim');

            if ($this->form_validation->run() === TRUE) {
                $data['name'] = $this->input->post('name');
                $data['bn_name'] = $this->input->post('bn_name');
                $data['note'] = $this->input->post('note');
                $data['sort_by'] = $this->input->post('sort_by');
                $data['status'] = $this->input->post('status') ? $this->input->post('status') : 0;
                $data['modified_at'] = date('Y-m-d H:i:s');
                $data['modified_by'] = logged_in_user_id();

                $updated = $this->diagnostic->update('diagnostic_departments', $data, array('id' => $this->input->post('id')));
                if ($updated) {
                    create_log('Has been updated a diagnostic department : ' . $data['name']);
                    success($this->lang->line('update_success'));
                    redirect('hospital/diagnostic/department');
                } else {
                    error($this->lang->line('update_failed'));
                    redirect('hospital/diagnostic/edit_department/' . $this->input->post('id'));
                }
            } else {
                $this->data['department'] = $this->diagnostic->get_single('diagnostic_departments', array('id' => $this->input->post('id')));
            }
        } else {
            if ($id) {
                $this->data['department'] = $this->diagnostic->get_single('diagnostic_departments', array('id' => $id));
                if (!$this->data['department']) {
                    redirect('hospital/diagnostic/department');
                }
            }
        }

        $this->data['edit'] = TRUE;
        $this->layout->title($this->lang->line('edit') . ' ' . $this->lang->line('department') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('diagnostic/department/edit', $this->data);
    }

    /*****************Function department_name**********************************
     * @type            : Function
     * @function name   : department_name
     * @description     : Unique check for "Department Name" data/value
     *
     * @param           : null
     * @return          : boolean true/false
     * ********************************************************** */
    public function department_name() {
        if ($this->input->post('id') == '') {
            $department = $this->diagnostic->duplicate_check_department_name($this->input->post('name'));
            if ($department) {
                $this->form_validation->set_message('department_name', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else if ($this->input->post('id') != '') {
            $department = $this->diagnostic->duplicate_check_department_name($this->input->post('name'), $this->input->post('id'));
            if ($department) {
                $this->form_validation->set_message('department_name', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else {
            return TRUE;
        }
    }

    /*****************Function delete_department**********************************
     * @type            : Function
     * @function name   : delete_department
     * @description     : delete "delete department" data from database
     *
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function delete_department($id = null) {

        check_permission(DELETE);

        if (!is_numeric($id)) {
            error($this->lang->line('unexpected_error'));
            redirect('hospital/diagnostic/department');
        }

        $department = $this->diagnostic->get_single('diagnostic_departments', array('id' => $id));

        if ($this->diagnostic->delete('diagnostic_departments', array('id' => $id))) {
            create_log('Has been deleted a diagnostic department : ' . $department->name);
            success($this->lang->line('delete_success'));
        } else {
            error($this->lang->line('delete_failed'));
        }
        redirect($_SERVER['HTTP_REFERER']);
    }

    /*****************Function group**********************************
     * @type            : Function
     * @function name   : group
     * @description     : Load "group List" user interface
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function group() {

        check_permission(VIEW);

        $search = $this->input->get('search');

        $config = array();
        $config = get_pagination(ADMIN_PER_PAGE);
        if ($search) {
            $config['base_url'] = site_url('hospital/diagnostic/group/?search=') . $search;
        } else {
            $config['base_url'] = current_url();
        }
        if ($this->input->get('page')) {
            $sgm = (int) trim($this->input->get('page'));
            $offset = $config['per_page'] * ($sgm - 1);
        } else {
            $offset = 0;
        }

        $config['total_rows'] = $this->diagnostic->get_group_list($config['per_page'], $offset, $search, $count = TRUE);
        $this->pagination->initialize($config);
        $this->data['groups'] = $this->diagnostic->get_group_list($config['per_page'], $offset, $search, $count = FALSE);
        $this->data['sl_count'] = ($this->input->get('page') == 0 ? 1 : (($this->input->get('page') - 1) * $config["per_page"] + 1));
        $this->data['search'] = $search;

        $this->data['list'] = TRUE;
        $this->layout->title($this->lang->line('manage_group') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('diagnostic/group/index', $this->data);
    }

    /*****************Function add_group**********************************
     * @type            : Function
     * @function name   : add_group
     * @description     : Load "Add new group" user interface
     *                    and process to store "diagnostic_groups" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function add_group() {

        check_permission(ADD);

        if ($_POST) {

            $this->load->library('form_validation');
            $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
            $this->form_validation->set_rules('department_id', $this->lang->line('department'), 'trim|required');
            $this->form_validation->set_rules('name', $this->lang->line('name'), 'trim|required|callback_group_name');
            $this->form_validation->set_rules('bn_name', $this->lang->line('bn_name'), 'trim');

            if ($this->form_validation->run() === TRUE) {
                $data['department_id'] = $this->input->post('department_id');
                $data['name'] = $this->input->post('name');
                $data['bn_name'] = $this->input->post('bn_name');
                $data['note'] = $this->input->post('note');
                $data['sort_by'] = $this->input->post('sort_by');
                $data['status'] = $this->input->post('status') ? $this->input->post('status') : 0;
                $data['created_at'] = date('Y-m-d H:i:s');
                $data['created_by'] = logged_in_user_id();

                $insert_id = $this->diagnostic->insert('diagnostic_groups', $data);
                if ($insert_id) {
                    create_log('Has been created a diagnostic group : ' . $data['name']);
                    success($this->lang->line('insert_success'));
                    redirect('hospital/diagnostic/group');
                } else {
                    error($this->lang->line('insert_failed'));
                    redirect('hospital/diagnostic/add_group');
                }
            } else {
                $this->data = $_POST;
            }
        }

        $this->data['departments'] = $this->diagnostic->get_list('diagnostic_departments', array('status' => 1), '', '', '', 'sort_by', 'ASC');
        $this->data['add'] = TRUE;
        $this->layout->title($this->lang->line('add') . ' ' . $this->lang->line('group') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('diagnostic/group/add', $this->data);
    }

    /*****************Function edit_group**********************************
     * @type            : Function
     * @function name   : edit_group
     * @description     : Load "Edit new group" user interface
     *                    and process to store "diagnostic_groups" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function edit_group($id = NULL) {

        check_permission(EDIT);

        if ($_POST) {

            $this->load->library('form_validation');
            $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
            $this->form_validation->set_rules('department_id', $this->lang->line('department'), 'trim|required');
            $this->form_validation->set_rules('name', $this->lang->line('name'), 'trim|required|callback_group_name');
            $this->form_validation->set_rules('bn_name', $this->lang->line('bn_name'), 'trim');

            if ($this->form_validation->run() === TRUE) {
                $data['department_id'] = $this->input->post('department_id');
                $data['name'] = $this->input->post('name');
                $data['bn_name'] = $this->input->post('bn_name');
                $data['note'] = $this->input->post('note');
                $data['sort_by'] = $this->input->post('sort_by');
                $data['status'] = $this->input->post('status') ? $this->input->post('status') : 0;
                $data['modified_at'] = date('Y-m-d H:i:s');
                $data['modified_by'] = logged_in_user_id();

                $updated = $this->diagnostic->update('diagnostic_groups', $data, array('id' => $this->input->post('id')));
                if ($updated) {
                    create_log('Has been updated a diagnostic group : ' . $data['name']);
                    success($this->lang->line('update_success'));
                    redirect('hospital/diagnostic/group');
                } else {
                    error($this->lang->line('update_failed'));
                    redirect('hospital/diagnostic/edit_group/' . $this->input->post('id'));
                }
            } else {
                $this->data['group'] = $this->diagnostic->get_single('diagnostic_groups', array('id' => $this->input->post('id')));
            }
        } else {
            if ($id) {
                $this->data['group'] = $this->diagnostic->get_single('diagnostic_groups', array('id' => $id));
                if (!$this->data['group']) {
                    redirect('hospital/diagnostic/group');
                }
            }
        }
        $this->data['departments'] = $this->diagnostic->get_list('diagnostic_departments', array('status' => 1), '', '', '', 'sort_by', 'ASC');
        $this->data['edit'] = TRUE;
        $this->layout->title($this->lang->line('edit') . ' ' . $this->lang->line('group') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('diagnostic/group/edit', $this->data);
    }

    /*****************Function group_name**********************************
     * @type            : Function
     * @function name   : group_name
     * @description     : Unique check for "Department Name" data/value
     *
     * @param           : null
     * @return          : boolean true/false
     * ********************************************************** */
    public function group_name() {
        if ($this->input->post('id') == '') {
            $group = $this->diagnostic->duplicate_check_group_name($this->input->post('department_id'), $this->input->post('name'));
            if ($group) {
                $this->form_validation->set_message('group_name', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else if ($this->input->post('id') != '') {
            $group = $this->diagnostic->duplicate_check_group_name($this->input->post('department_id'), $this->input->post('name'), $this->input->post('id'));
            if ($group) {
                $this->form_validation->set_message('group_name', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else {
            return TRUE;
        }
    }

    /*****************Function delete_group**********************************
     * @type            : Function
     * @function name   : delete_group
     * @description     : delete "delete group" data from database
     *
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function delete_group($id = null) {

        check_permission(DELETE);

        if (!is_numeric($id)) {
            error($this->lang->line('unexpected_error'));
            redirect('hospital/diagnostic/group');
        }

        $group = $this->diagnostic->get_single('diagnostic_groups', array('id' => $id));

        if ($this->diagnostic->delete('diagnostic_groups', array('id' => $id))) {
            create_log('Has been deleted a diagnostic group : ' . $group->name);
            success($this->lang->line('delete_success'));
        } else {
            error($this->lang->line('delete_failed'));
        }
        redirect($_SERVER['HTTP_REFERER']);
    }


    /*****************Function test**********************************
     * @type            : Function
     * @function name   : test
     * @description     : Load "test List" user interface
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function test() {

        check_permission(VIEW);

        $department_id = $this->input->get('department_id');
        $group_id = $this->input->get('group_id');
        $search = trim($this->input->get('search'));

        $config = array();
        $config = get_pagination(ADMIN_PER_PAGE);
        if ($department_id || $group_id || $search) {
            $config['base_url'] = site_url('hospital/diagnostic/test/?department_id=' . $department_id . '&group_id=' . $group_id . '&search=' . $search);
        } else {
            $config['base_url'] = current_url();
        }
        if ($this->input->get('page')) {
            $sgm = (int) trim($this->input->get('page'));
            $offset = $config['per_page'] * ($sgm - 1);
        } else {
            $offset = 0;
        }

        $config['total_rows'] = $this->diagnostic->get_test_list($config['per_page'], $offset, $department_id, $group_id, $search, $count = TRUE);
        $this->pagination->initialize($config);
        $this->data['tests'] = $this->diagnostic->get_test_list($config['per_page'], $offset, $department_id, $group_id, $search, $count = FALSE);
        $this->data['sl_count'] = ($this->input->get('page') == 0 ? 1 : (($this->input->get('page') - 1) * $config["per_page"] + 1));
        $this->data['department_id'] = $department_id;
        $this->data['group_id'] = $group_id;
        $this->data['search'] = $search;

        $this->data['departments'] = $this->diagnostic->get_list('diagnostic_departments', array('status' => 1), '', '', '', 'sort_by', 'ASC');
        $this->data['groups'] = $this->diagnostic->get_list('diagnostic_groups', array('status' => 1), '', '', '', 'sort_by', 'ASC');
        $this->data['list'] = TRUE;
        $this->layout->title($this->lang->line('diagnostic') . ' ' . $this->lang->line('test') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('diagnostic/test/index', $this->data);
    }

    /*****************Function add_test**********************************
     * @type            : Function
     * @function name   : add_test
     * @description     : Load "Add new test" user interface
     *                    and process to store "diagnostic_tests" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function add_test() {

        check_permission(ADD);

        if ($_POST) {

            $this->load->library('form_validation');
            $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
            $this->form_validation->set_rules('department_id', $this->lang->line('department'), 'trim|required');
            $this->form_validation->set_rules('name', $this->lang->line('name'), 'trim|required|callback_test_name');

            if ($this->form_validation->run() === TRUE) {
                $data['department_id'] = $this->input->post('department_id');
                $data['group_id'] = $this->input->post('group_id');
                $data['tube_id'] = $this->input->post('tube_id');
                $data['needle_id'] = $this->input->post('needle_id');
                $data['specimen'] = $this->input->post('specimen');
                $data['medicine_id'] = $this->input->post('medicine_id');
                $data['name'] = $this->input->post('name');
                $data['code'] = $this->input->post('code');
                $data['room_no'] = $this->input->post('room_no');
                $data['rate'] = $this->input->post('rate');
                $data['special_discount'] = $this->input->post('special_discount');
                $data['mbbs'] = $this->input->post('mbbs');
                $data['vd'] = $this->input->post('vd');
                $data['emp'] = $this->input->post('emp');
                $data['corporate'] = $this->input->post('corporate');
                $data['note'] = $this->input->post('note');
                $data['sort_by'] = $this->input->post('sort_by');
                $data['status'] = $this->input->post('status') ? $this->input->post('status') : 0;
                $data['created_at'] = date('Y-m-d H:i:s');
                $data['created_by'] = logged_in_user_id();

                $insert_id = $this->diagnostic->insert('diagnostic_tests', $data);
                if ($insert_id) {
                    create_log('Has been created a diagnostic test : ' . $data['name']);
                    success($this->lang->line('insert_success'));
                    redirect('hospital/diagnostic/test');
                } else {
                    error($this->lang->line('insert_failed'));
                    redirect('hospital/diagnostic/add_test');
                }
            } else {
                $this->data = $_POST;
            }
        }

        $this->data['departments'] = $this->diagnostic->get_list('diagnostic_departments', array('status' => 1), '', '', '', 'sort_by', 'ASC');
        $this->data['tubes'] = $this->diagnostic->get_list('diagnostic_tubes', array('status' => 1), '', '', '', 'sort_by', 'ASC');
        $this->data['add'] = TRUE;
        $this->layout->title($this->lang->line('add') . ' ' . $this->lang->line('test') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('diagnostic/test/add', $this->data);
    }

    /*****************Function edit_test**********************************
     * @type            : Function
     * @function name   : edit_test
     * @description     : Load "Edit new test" user interface
     *                    and process to store "diagnostic_tests" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function edit_test($id = NULL) {

        check_permission(EDIT);

        if ($_POST) {

            $this->load->library('form_validation');
            $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
            $this->form_validation->set_rules('department_id', $this->lang->line('department'), 'trim|required');
            $this->form_validation->set_rules('name', $this->lang->line('name'), 'trim|required|callback_test_name');

            if ($this->form_validation->run() === TRUE) {
                $data['department_id'] = $this->input->post('department_id');
                $data['group_id'] = $this->input->post('group_id');
                $data['tube_id'] = $this->input->post('tube_id');
                $data['needle_id'] = $this->input->post('needle_id');
                $data['specimen'] = $this->input->post('specimen');
                $data['medicine_id'] = $this->input->post('medicine_id');
                $data['name'] = $this->input->post('name');
                $data['code'] = $this->input->post('code');
                $data['room_no'] = $this->input->post('room_no');
                $data['rate'] = $this->input->post('rate');
                $data['special_discount'] = $this->input->post('special_discount');
                $data['mbbs'] = $this->input->post('mbbs');
                $data['vd'] = $this->input->post('vd');
                $data['emp'] = $this->input->post('emp');
                $data['corporate'] = $this->input->post('corporate');
                $data['note'] = $this->input->post('note');
                $data['sort_by'] = $this->input->post('sort_by');
                $data['status'] = $this->input->post('status') ? $this->input->post('status') : 0;
                $data['modified_at'] = date('Y-m-d H:i:s');
                $data['modified_by'] = logged_in_user_id();

                $updated = $this->diagnostic->update('diagnostic_tests', $data, array('id' => $this->input->post('id')));
                if ($updated) {
                    create_log('Has been updated a diagnostic test : ' . $data['name']);
                    success($this->lang->line('update_success'));
                    redirect('hospital/diagnostic/test');
                } else {
                    error($this->lang->line('update_failed'));
                    redirect('hospital/diagnostic/edit_test/' . $this->input->post('id'));
                }
            } else {
                $this->data['test'] = $this->diagnostic->get_single('diagnostic_tests', array('id' => $this->input->post('id')));
            }
        } else {
            if ($id) {
                $this->data['test'] = $this->diagnostic->get_single('diagnostic_tests', array('id' => $id));
                if (!$this->data['test']) {
                    redirect('hospital/diagnostic/test');
                }
            }
        }
        $this->data['departments'] = $this->diagnostic->get_list('diagnostic_departments', array('status' => 1), '', '', '', 'sort_by', 'ASC');
        $this->data['groups'] = $this->diagnostic->get_list('diagnostic_groups', array('status' => 1), '', '', '', 'sort_by', 'ASC');
        $this->data['tubes'] = $this->diagnostic->get_list('diagnostic_tubes', array('status' => 1), '', '', '', 'sort_by', 'ASC');
        $this->data['edit'] = TRUE;
        $this->layout->title($this->lang->line('edit') . ' ' . $this->lang->line('test') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('diagnostic/test/edit', $this->data);
    }

    /*     * ***************Function get_single_test**********************************
     * @type            : Function
     * @function name   : get_single_test
     * @description     : "Load single test information" from database                  
     *                    to the user interface   
     * @param           : null
     * @return          : null 
     * ********************************************************** */

    public function get_single_test() {
        $id = $this->input->post('id');
        $this->data['test'] = $this->diagnostic->get_single_test($id);
        echo $this->load->view('diagnostic/test/view', $this->data);
    }

    /*****************Function test_name**********************************
     * @type            : Function
     * @function name   : test_name
     * @description     : Unique check for "Department Name" data/value
     *
     * @param           : null
     * @return          : boolean true/false
     * ********************************************************** */
    public function test_name() {
        if ($this->input->post('id') == '') {
            $test = $this->diagnostic->duplicate_check_test_name($this->input->post('department_id'), $this->input->post('name'));
            if ($test) {
                $this->form_validation->set_message('test_name', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else if ($this->input->post('id') != '') {
            $test = $this->diagnostic->duplicate_check_test_name($this->input->post('department_id'), $this->input->post('name'), $this->input->post('id'));
            if ($test) {
                $this->form_validation->set_message('test_name', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else {
            return TRUE;
        }
    }

    /*****************Function delete_test**********************************
     * @type            : Function
     * @function name   : delete_test
     * @description     : delete "delete test" data from database
     *
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function delete_test($id = null) {

        check_permission(DELETE);

        if (!is_numeric($id)) {
            error($this->lang->line('unexpected_error'));
            redirect('hospital/diagnostic/test');
        }

        $test = $this->diagnostic->get_single('diagnostic_tests', array('id' => $id));

        if ($this->diagnostic->delete('diagnostic_tests', array('id' => $id))) {
            create_log('Has been deleted a diagnostic test : ' . $test->name);
            success($this->lang->line('delete_success'));
        } else {
            error($this->lang->line('delete_failed'));
        }
        redirect($_SERVER['HTTP_REFERER']);
    }

    /*****************Function price**********************************
     * @type            : Function
     * @function name   : price
     * @description     : Load "Test Price Update" user interface
     *                    and process to store "diagnostic_tests" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function price() {

        check_permission(ADD);

        if ($_POST) {
            if (!empty($_POST['test_id'])) {
                foreach ($_POST['test_id'] as $key => $value) {
                    $id = $value;
                    $data['code'] = $_POST['code'][$value];
                    $data['room_no'] = $_POST['room_no'][$value];
                    $data['specimen'] = $_POST['specimen'][$value];
                    $data['rate'] = $_POST['rate'][$value];
                    $data['special_discount'] = $_POST['special_discount'][$value];
                    $data['mbbs'] = $_POST['mbbs'][$value];
                    $data['vd'] = $_POST['vd'][$value];
                    $data['emp'] = $_POST['emp'][$value];
                    $data['corporate'] = $_POST['corporate'][$value];

                    $data['modified_at'] = date('Y-m-d H:i:s');
                    $data['modified_by'] = logged_in_user_id();
                    $updated_id =  $this->diagnostic->update('diagnostic_tests', $data, array('id' => $id));
                }

                if ($updated_id) {
                    $user =  $this->diagnostic->get_single('employees', array('status' => 1, 'user_id' => $this->session->userdata('user_id')));
                    create_log('Has been updated a diagnostic test for multiple data by : ' . $user->name);
                    success($this->lang->line('update_success'));
                    redirect('hospital/diagnostic/price');
                } else {
                    error($this->lang->line('update_failed'));
                    redirect('hospital/diagnostic/price');
                }
            }
        }

        $department_id = $this->input->get('department_id');
        $group_id = $this->input->get('group_id');

        if ($_GET || $department_id || $group_id) {
            $this->data['tests'] =  $this->diagnostic->get_test_price_update_list($department_id, $group_id);

            $this->data['department_id'] = $department_id;
            $this->data['group_id'] = $group_id;
        }

        $this->data['departments'] = $this->diagnostic->get_list('diagnostic_departments', array('status' => 1), '', '', '', 'id', 'ASC');
        $this->data['groups'] = $this->diagnostic->get_list('	diagnostic_groups', array('status' => 1), '', '', '', 'id', 'ASC');

        $this->data['add'] = TRUE;
        $this->layout->title($this->lang->line('price') . ' ' . $this->lang->line('update') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('diagnostic/test/price', $this->data);
    }

    /*****************Function tube**********************************
     * @type            : Function
     * @function name   : tube
     * @description     : Load "tube List" user interface
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function tube() {

        check_permission(VIEW);

        $search = $this->input->get('search');

        $config = array();
        $config = get_pagination(ADMIN_PER_PAGE);
        if ($search) {
            $config['base_url'] = site_url('hospital/diagnostic/tube/?search=') . $search;
        } else {
            $config['base_url'] = current_url();
        }
        if ($this->input->get('page')) {
            $sgm = (int) trim($this->input->get('page'));
            $offset = $config['per_page'] * ($sgm - 1);
        } else {
            $offset = 0;
        }

        $config['total_rows'] = $this->diagnostic->get_tube_list($config['per_page'], $offset, $search, $count = TRUE);
        $this->pagination->initialize($config);
        $this->data['tubes'] = $this->diagnostic->get_tube_list($config['per_page'], $offset, $search, $count = FALSE);
        $this->data['sl_count'] = ($this->input->get('page') == 0 ? 1 : (($this->input->get('page') - 1) * $config["per_page"] + 1));
        $this->data['search'] = $search;

        $this->data['list'] = TRUE;
        $this->layout->title($this->lang->line('diagnostic') . ' ' . $this->lang->line('tube') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('diagnostic/tube/index', $this->data);
    }

    /*****************Function add_tube**********************************
     * @type            : Function
     * @function name   : add_tube
     * @description     : Load "Add new tube" user interface
     *                    and process to store "diagnostic_tubes" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function add_tube() {

        check_permission(ADD);

        if ($_POST) {

            $this->load->library('form_validation');
            $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
            $this->form_validation->set_rules('name', $this->lang->line('name'), 'trim|required');
            $this->form_validation->set_rules('price', $this->lang->line('price'), 'trim|required');

            if ($this->form_validation->run() === TRUE) {
                $data['name'] = $this->input->post('name');
                $data['type'] = $this->input->post('type');
                $data['price'] = $this->input->post('price');
                $data['note'] = $this->input->post('note');
                $data['sort_by'] = $this->input->post('sort_by');
                $data['status'] = $this->input->post('status') ? $this->input->post('status') : 0;
                $data['created_at'] = date('Y-m-d H:i:s');
                $data['created_by'] = logged_in_user_id();

                $insert_id = $this->diagnostic->insert('diagnostic_tubes', $data);
                if ($insert_id) {
                    create_log('Has been created a diagnostic tube : ' . $data['name']);
                    success($this->lang->line('insert_success'));
                    redirect('hospital/diagnostic/tube');
                } else {
                    error($this->lang->line('insert_failed'));
                    redirect('hospital/diagnostic/add_tube');
                }
            } else {
                $this->data = $_POST;
            }
        }

        $this->data['add'] = TRUE;
        $this->layout->title($this->lang->line('add') . ' ' . $this->lang->line('tube') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('diagnostic/tube/add', $this->data);
    }

    /*****************Function edit_tube**********************************
     * @type            : Function
     * @function name   : edit_tube
     * @description     : Load "Edit new tube" user interface
     *                    and process to store "diagnostic_tubes" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function edit_tube($id = NULL) {

        check_permission(EDIT);

        if ($_POST) {

            $this->load->library('form_validation');
            $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
            $this->form_validation->set_rules('name', $this->lang->line('name'), 'trim|required');
            $this->form_validation->set_rules('price', $this->lang->line('price'), 'trim|required');

            if ($this->form_validation->run() === TRUE) {
                $data['name'] = $this->input->post('name');
                $data['type'] = $this->input->post('type');
                $data['price'] = $this->input->post('price');
                $data['note'] = $this->input->post('note');
                $data['sort_by'] = $this->input->post('sort_by');
                $data['status'] = $this->input->post('status') ? $this->input->post('status') : 0;
                $data['modified_at'] = date('Y-m-d H:i:s');
                $data['modified_by'] = logged_in_user_id();

                $updated = $this->diagnostic->update('diagnostic_tubes', $data, array('id' => $this->input->post('id')));
                if ($updated) {
                    create_log('Has been updated a diagnostic tube : ' . $data['name']);
                    success($this->lang->line('update_success'));
                    redirect('hospital/diagnostic/tube');
                } else {
                    error($this->lang->line('update_failed'));
                    redirect('hospital/diagnostic/edit_tube/' . $this->input->post('id'));
                }
            } else {
                $this->data['tube'] = $this->diagnostic->get_single('diagnostic_tubes', array('id' => $this->input->post('id')));
            }
        } else {
            if ($id) {
                $this->data['tube'] = $this->diagnostic->get_single('diagnostic_tubes', array('id' => $id));
                if (!$this->data['tube']) {
                    redirect('hospital/diagnostic/tube');
                }
            }
        }
        $this->data['edit'] = TRUE;
        $this->layout->title($this->lang->line('edit') . ' ' . $this->lang->line('tube') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('diagnostic/tube/edit', $this->data);
    }

    /*     * ***************Function get_single_tube**********************************
     * @type            : Function
     * @function name   : get_single_tube
     * @description     : "Load single tube information" from database                  
     *                    to the user interface   
     * @param           : null
     * @return          : null 
     * ********************************************************** */

    public function get_single_tube() {
        $id = $this->input->post('id');
        $this->data['tube'] = $this->diagnostic->get_single_tube($id);
        echo $this->load->view('diagnostic/tube/view', $this->data);
    }

    /*****************Function delete_tube**********************************
     * @type            : Function
     * @function name   : delete_tube
     * @description     : delete "delete tube" data from database
     *
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function delete_tube($id = null) {

        check_permission(DELETE);

        if (!is_numeric($id)) {
            error($this->lang->line('unexpected_error'));
            redirect('hospital/diagnostic/tube');
        }

        $tube = $this->diagnostic->get_single('diagnostic_tubes', array('id' => $id));

        if ($this->diagnostic->delete('diagnostic_tubes', array('id' => $id))) {
            create_log('Has been deleted a diagnostic tube : ' . $tube->name);
            success($this->lang->line('delete_success'));
        } else {
            error($this->lang->line('delete_failed'));
        }
        redirect($_SERVER['HTTP_REFERER']);
    }
}
