<?php

defined('BASEPATH') or exit('No direct script access allowed');

/* * *****************Service.php**********************************
 * @product name    : Hospital Management System
 * @type            : Class
 * @class name      : Service
 * @description     : Manage Service frontend website. 
 * @author          : South Bangla IT Zone Team
 * @url             : http://hms.sbitzone.com
 * @support         : rabibd.it@gmail.com
 * @copyright       : South Bangla IT Zone Team
 * ********************************************************** */

class Service extends MY_Controller {

    public $data = array();

    function __construct() {
        parent::__construct();
        $this->load->model('Service_Model', 'service', true);
    }

    /*****************Function index**********************************
     * @type            : Function
     * @function name   : index
     * @description     : Load "Service List" user interface
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function index() {

        check_permission(VIEW);

        $keyword = $this->input->get('keyword');
        $config = array();
        $config = get_pagination(ADMIN_PER_PAGE);
        if ($keyword) {
            $config['base_url'] = site_url('frontend/service/?keyword=' . $keyword);
        } else {
            $config['base_url'] = current_url();
        }
        if ($this->input->get('page')) {
            $sgm = (int) trim($this->input->get('page'));
            $offset = $config['per_page'] * ($sgm - 1);
        } else {
            $offset = 0;
        }

        $config['total_rows'] = $this->service->get_service_list($config['per_page'], $offset, $keyword, $count = TRUE);
        $this->pagination->initialize($config);
        $this->data['services'] = $this->service->get_service_list($config['per_page'], $offset, $keyword, $count = FALSE);
        $this->data['sl_count'] = ($this->input->get('page') == 0 ? 1 : (($this->input->get('page') - 1) * $config["per_page"] + 1));
        $this->data['keyword'] = $keyword;

        $this->data['list'] = TRUE;
        $this->layout->title($this->lang->line('manage_service') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('service/index', $this->data);
    }

    /*****************Function add**********************************
     * @type            : Function
     * @function name   : add
     * @description     : Load "Add new Service" user interface
     *                    and process to store "Service" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function add() {

        check_permission(ADD);

        if ($_POST) {
            $this->_prepare_service_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_service_data();
                $insert_id = $this->service->insert('services', $data);
                if ($insert_id) {
                    create_log('Has been created a service : ' . $data['title']);
                    success($this->lang->line('insert_success'));
                    redirect('frontend/service');
                } else {
                    error($this->lang->line('insert_failed'));
                    redirect('frontend/service/add');
                }
            } else {
                $this->data = $_POST;
            }
        }

        $this->data['add'] = TRUE;
        $this->layout->title($this->lang->line('add') . ' ' . $this->lang->line('service') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('service/add', $this->data);
    }

    /*****************Function edit**********************************
     * @type            : Function
     * @function name   : edit
     * @description     : Load Update "Service" user interface
     *                    with populate "Service" value
     *                    and process to update "Service" into database
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function edit($id = null) {

        check_permission(EDIT);

        if ($_POST) {
            $this->_prepare_service_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_service_data();
                $updated = $this->service->update('services', $data, array('id' => $this->input->post('id')));
                if ($updated) {
                    create_log('Has been updated a service : ' . $data['title']);
                    success($this->lang->line('update_success'));
                    redirect('frontend/service');
                } else {
                    error($this->lang->line('update_failed'));
                    redirect('frontend/service/edit/' . $this->input->post('id'));
                }
            } else {
                $this->data['service'] = $this->service->get_single('services', array('id' => $this->input->post('id')));
            }
        } else {
            if ($id) {
                $this->data['service'] = $this->service->get_single('services', array('id' => $id));
                if (!$this->data['service']) {
                    redirect('frontend/service');
                }
            }
        }

        $this->data['edit'] = TRUE;
        $this->layout->title($this->lang->line('edit') . ' ' . $this->lang->line('service') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('service/edit', $this->data);
    }

    /*     * ***************Function _prepare_service_validation**********************************
     * @type            : Function
     * @function name   : _prepare_service_validation
     * @description     : Process "service" user input data validation                 
     *                       
     * @param           : null
     * @return          : null 
     * ********************************************************** */

    private function _prepare_service_validation() {
        $this->load->library('form_validation');
        $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
        $this->form_validation->set_rules('type', $this->lang->line('type'), 'trim|required');
        $this->form_validation->set_rules('title', $this->lang->line('title'), 'trim|required');
        $this->form_validation->set_rules('image', $this->lang->line('service') . ' ' . $this->lang->line('image'), 'trim|callback_image');
    }

    /*     * ***************Function image**********************************
     * @type            : Function
     * @function name   : image
     * @description     : validate  service  image type/format                  
     *                       
     * @param           : null
     * @return          : boolean true/false 
     * ********************************************************** */

    public function image() {

        if ($this->input->post('id')) {
            if (!empty($_FILES['image']['name'])) {
                $name = $_FILES['image']['name'];
                $arr = explode('.', $name);
                $ext = end($arr);
                if ($ext == 'jpg' || $ext == 'jpeg' || $ext == 'png' || $ext == 'gif') {
                    return TRUE;
                } else {
                    $this->form_validation->set_message('image', $this->lang->line('select_valid_file_format'));
                    return FALSE;
                }
            }
        } else {
            if (isset($_FILES['image']['name'])) {
                $name = $_FILES['image']['name'];
                $arr = explode('.', $name);
                $ext = end($arr);
                if ($ext == 'jpg' || $ext == 'jpeg' || $ext == 'png' || $ext == 'gif') {
                    return TRUE;
                } else {
                    $this->form_validation->set_message('image', $this->lang->line('select_valid_file_format'));
                    return FALSE;
                }
            } else {
                $this->form_validation->set_message('image', $this->lang->line('required_field'));
                return FALSE;
            }
        }
    }

    /*     * ***************Function _get_posted_service_data**********************************
     * @type            : Function
     * @function name   : _get_posted_service_data
     * @description     : Prepare "service" user input data to save into database                  
     *                       
     * @param           : null
     * @return          : $data array(); value 
     * ********************************************************** */

    private function _get_posted_service_data() {

        $items = array();
        $items[] = 'type';
        $items[] = 'title';
        $items[] = 'bn_title';
        $items[] = 'description';
        $items[] = 'bn_description';
        $items[] = 'icon';
        $items[] = 'sort_by';

        $data = elements($items, $_POST);

        $data['is_view_on_home'] = $this->input->post('is_view_on_home') ?? 0;
        $data['is_view_on_web'] = $this->input->post('is_view_on_web') ?? 0;

        if ($this->input->post('id')) {
            $data['modified_at'] = date('Y-m-d H:i:s');
            $data['modified_by'] = logged_in_user_id();
        } else {
            $data['status'] = 1;
            $data['created_at'] = date('Y-m-d H:i:s');
            $data['created_by'] = logged_in_user_id();
        }

        if (isset($_FILES['image']['name'])) {
            $data['image'] = $this->_upload_image();
        }

        return $data;
    }

    /*     * ***************Function _upload_service**********************************
     * @type            : Function
     * @function name   : _upload_service
     * @description     : Process to to upload service in the server
     *                    and return service name                   
     *                       
     * @param           : null
     * @return          : $return_service string value 
     * ********************************************************** */

    private function _upload_image() {

        $prev_image = $this->input->post('prev_image');
        $image = $_FILES['image']['name'];
        $image_type = $_FILES['image']['type'];
        $return_image = '';
        if ($image != "") {
            if (
                $image_type == 'image/jpeg' || $image_type == 'image/pjpeg' ||
                $image_type == 'image/jpg' || $image_type == 'image/png' ||
                $image_type == 'image/x-png' || $image_type == 'image/gif'
            ) {

                $destination = 'assets/uploads/service-image/';

                $file_type = explode(".", $image);
                $extension = strtolower($file_type[count($file_type) - 1]);
                $image_path = 'service-image' . time() . '-sbiz.' . $extension;

                move_uploaded_file($_FILES['image']['tmp_name'], $destination . $image_path);

                // need to unlink previous service
                if ($prev_image != "") {
                    if (file_exists($destination . $prev_image)) {
                        @unlink($destination . $prev_image);
                    }
                }

                $return_image = $image_path;
            }
        } else {
            $return_image = $prev_image;
        }

        return $return_image;
    }

    /*     * ***************Function delete**********************************
     * @type            : Function
     * @function name   : delete
     * @description     : delete "service" from database                  
     *                    and unlink service from server   
     * @param           : $id integer value
     * @return          : null 
     * ********************************************************** */

    public function delete($id = null) {

        check_permission(DELETE);

        if (!is_numeric($id)) {
            error($this->lang->line('unexpected_error'));
            redirect('frontend/service');
        }

        $service = $this->service->get_single('services', array('id' => $id));
        if ($this->service->delete('services', array('id' => $id))) {
            // delete service
            $destination = 'assets/uploads/';
            if (file_exists($destination . '/service-image-image/' . $service->image)) {
                @unlink($destination . '/service/' . $service->image);
            }
            create_log('Has been deletd a service : ' . $service->title);
            success($this->lang->line('delete_success'));
        } else {
            error($this->lang->line('delete_failed'));
        }
        redirect($_SERVER['HTTP_REFERER']);
    }
}
