<?php

defined('BASEPATH') or exit('No direct script access allowed');

/* * *****************Category.php**********************************
 * @product name    : Hospital Management System
 * @type            : Class
 * @class name      : Category
 * @description     : Manage category.
 * @author          : South Bangla IT Zone Team
 * @url             : http://hms.sbitzone.com
 * @support         : rabibd.it@gmail.com
 * @copyright       : South Bangla IT Zone Team
 * ********************************************************** */

class Category extends MY_Controller
{

    public $data = array();

    function __construct()
    {
        parent::__construct();
        $this->load->model('Category_Model', 'category', true);
    }

    /*****************Function index**********************************
     * @type            : Function
     * @function name   : index
     * @description     : Load "Category List" user interface
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function index()
    {

        check_permission(VIEW);

        $config = array();
        $config = get_pagination(ADMIN_PER_PAGE);
        $config['base_url'] = current_url();
        if ($this->input->get('page')) {
            $sgm = (int) trim($this->input->get('page'));
            $offset = $config['per_page'] * ($sgm - 1);
        } else {
            $offset = 0;
        }

        $config['total_rows'] = $this->category->get_category_list($config['per_page'], $offset, $count = TRUE);
        $this->pagination->initialize($config);
        $this->data['categories'] = $this->category->get_category_list($config['per_page'], $offset, $count = FALSE);
        $this->data['sl_count'] = ($this->input->get('page') == 0 ? 1 : (($this->input->get('page') - 1) * $config["per_page"] + 1));

        $this->data['list'] = TRUE;
        $this->layout->title($this->lang->line('manage_category') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('category/index', $this->data);
    }

    /*****************Function add**********************************
     * @type            : Function
     * @function name   : add
     * @description     : Load "Add new Category" user interface
     *                    and process to store "followup_categories" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function add()
    {

        check_permission(ADD);

        if ($_POST) {
            $this->_prepare_category_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_category_data();
                $insert_id = $this->category->insert('followup_categories', $data);
                if ($insert_id) {
                    create_log('Has been created a Category : ' . $data['name']);
                    success($this->lang->line('insert_success'));
                    redirect('followup/category');
                } else {
                    error($this->lang->line('insert_failed'));
                    redirect('followup/category/add');
                }
            } else {
                $this->data = $_POST;
            }
        }

        $this->data['add'] = TRUE;
        $this->layout->title($this->lang->line('add') . ' ' . $this->lang->line('category') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('category/add', $this->data);
    }

    /*****************Function edit**********************************
     * @type            : Function
     * @function name   : edit
     * @description     : Load Update "Category" user interface
     *                    with populate "Category" value
     *                    and process to update "followup_categories" into database
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function edit($id = null)
    {

        check_permission(EDIT);

        if ($_POST) {
            $this->_prepare_category_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_category_data();
                $updated = $this->category->update('followup_categories', $data, array('id' => $this->input->post('id')));
                if ($updated) {
                    create_log('Has been updated a Category : ' . $data['name']);
                    success($this->lang->line('update_success'));
                    redirect('followup/category');
                } else {
                    error($this->lang->line('update_failed'));
                    redirect('followup/category/edit/' . $this->input->post('id'));
                }
            } else {
                $this->data['category'] = $this->category->get_single('followup_categories', array('id' => $this->input->post('id')));
            }
        } else {
            if ($id) {
                $this->data['category'] = $this->category->get_single('followup_categories', array('id' => $id));
                if (!$this->data['category']) {
                    redirect('followup/category');
                }
            }
        }

        $this->data['edit'] = TRUE;
        $this->layout->title($this->lang->line('edit') . ' ' . $this->lang->line('category') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('category/edit', $this->data);
    }


    /*     * ***************Function get_single_category**********************************
     * @type            : Function
     * @function name   : get_single_category
     * @description     : "Load single category information" from database                  
     *                    to the user interface   
     * @param           : null
     * @return          : null 
     * ********************************************************** */

    public function get_single_category()
    {
        $id = $this->input->post('id');
        $this->data['category'] = $this->category->get_single_category($id);
        echo $this->load->view('category/view', $this->data);
    }

    /*****************Function _prepare_category_validation**********************************
     * @type            : Function
     * @function name   : _prepare_category_validation
     * @description     : Process "Category" user input data validation
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    private function _prepare_category_validation()
    {
        $this->load->library('form_validation');
        $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
        $this->form_validation->set_rules('name', $this->lang->line('name'), 'trim|required');
        $this->form_validation->set_rules('day', $this->lang->line('day'), 'trim|required');
    }

    /*****************Function _get_posted_category_data**********************************
     * @type            : Function
     * @function name   : _get_posted_category_data
     * @description     : Prepare "Category" user input data to save into database
     *
     * @param           : null
     * @return          : $data array(); value
     * ********************************************************** */
    private function _get_posted_category_data()
    {

        $items = array();
        $items[] = 'name';
        $items[] = 'day';
        $items[] = 'note';
        $items[] = 'sort_by';
        $items[] = 'status';
        $data = elements($items, $_POST);

        if ($this->input->post('id')) {
            $data['modified_at'] = date('Y-m-d H:i:s');
            $data['modified_by'] = logged_in_user_id();
        } else {
            $data['created_at'] = date('Y-m-d H:i:s');
            $data['created_by'] = logged_in_user_id();
        }

        return $data;
    }


    /*****************Function delete**********************************
     * @type            : Function
     * @function name   : delete
     * @description     : delete "Category" data from database
     *
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function delete($id = null)
    {

        check_permission(DELETE);

        if (!is_numeric($id)) {
            error($this->lang->line('unexpected_error'));
            redirect('followup/category');
        }

        $category = $this->category->get_single('followup_categories', array('id' => $id));

        if ($this->category->delete('followup_categories', array('id' => $id))) {
            create_log('Has been deleted a Category : ' . $category->name);
            success($this->lang->line('delete_success'));
        } else {
            error($this->lang->line('delete_failed'));
        }
        redirect($_SERVER['HTTP_REFERER']);
    }
}
